PRO MUSE_ETC, Action=Action, Flux=flux, AB=AB, Wave=Wave, Texp=Texp, Nexp=Nexp, Source=Source, $
             Atm=Atm, Airmass=Airmass, Mode=Mode, sn=sn, nspec=nspec, nspat=nspat, quiet=quiet, $
             help=help

Common Share, lbda, l, lmu, Extinct, Tmuse, FSky, flag_AB, hc, VLT, DS, RN, DC, DA, Disp

version = '1.4 - 17/01/04'

; F is the object flux in erg/s/cm2 (by A if continuum source)
; if AB is set, then F is in AB magnitude
; Lbda is the wavelength in A (defaut I band 7900 A)
; Texp is the integration time in sec of one exposure (default 1 hour)
; Nexp is the number of exposure (defaul 1)
; Source is the source type: PC, EC, PL, EL (defaulted to PC)
;         with P=Ponctual (spatially unresolved), E=Extended (spatially resolved)
;              L=Line emission (unresolved), C=Continuum (flat continuum source)
; Atm is the atmosperic conditions: NP, NG, AP, AG (defaulted to AG)
;         with N=NoAO A=AO P=Poor G=Good
; Airmass is the observation airmass (defaulted to 1)
; Mode is the MUSE mode: WFM or NFM (defaulted to WFM)
; nspec is the number of spectral pixel to sum up (defaulted to 1), only in continuum
; nspat is the number of spatial pixel to sum up (defaulted to 1), only in extended
; quiet, if set no print and no plots

if ~Keyword_set(quiet) then print,'muse_etc.pro version', version

if Keyword_set(help) then begin
    print,'Usage muse_pro, action=S|F|T, flux=flux, sn=sn, texp=texp, /ab, wave=7900, nexp=1, source=pc|pl|ec|el'
    print,'               atm=ag|ap|ng|np, airmass=1, mode=WFM|NFM, nspec=1, nspat=1'
    return
endif

THROUGHPUT = 'T' ; typical throughput WFM

if ~Keyword_set(Action) then Action = 'S' ; default action = Compute S/N
if ~Keyword_set(Flux) then Flux = 1.e-18 ; default flux
if ~Keyword_set(SN) then SN = 5.0 ; default S/N
if ~Keyword_set(Texp) then Texp = 3600.0 ; default Integration time in sec
if ~Keyword_set(Wave) then Wave = 7900.0 ; default wavelength (I band)
if ~Keyword_set(Nexp) then Nexp = 1 ; default number of exposure
if ~Keyword_set(Source) then Source = 'PC' ; default source type
if ~Keyword_set(Atm) then Atm = 'AG' ; default atmosphere parameter
if ~Keyword_set(Airmass) then Airmass = 1.0 ; default airmass (zenith)
if ~Keyword_set(nspec) then nspec = 1 ; default number of spectrum pixel to sum up
if ~Keyword_set(nspat) then nspat = 1 ; default number of spatial pixel to sum up
if ~Keyword_set(Mode) then Mode = 'WFM' ; default instrument mode WFM
flag_AB = 0
if Keyword_set(AB) then flag_AB = 1 ; check if AB magnitude flag is set

; check which mode we are

Case Action of
    'S': begin
           OnlyOne, wave, flux, texp
           if (n_elements(wave) gt 1) then sn = fltarr(n_elements(wave))
           if (n_elements(flux) gt 1) then sn = fltarr(n_elements(flux))
           if (n_elements(texp) gt 1) then sn = fltarr(n_elements(texp))
           if ~Keyword_set(quiet) then print, 'Compute S/N'
         end
    'T': begin
           OnlyOne, wave, flux, sn
           if (n_elements(wave) gt 1) then texp = fltarr(n_elements(wave))
           if (n_elements(flux) gt 1) then texp = fltarr(n_elements(flux))
           if (n_elements(sn) gt 1) then texp = fltarr(n_elements(sn))
           if ~Keyword_set(quiet) then print, 'Compute Exposure time'
         end
    'F': begin
           OnlyOne, wave, texp, sn
           if (n_elements(wave) gt 1) then flux = fltarr(n_elements(wave))
           if (n_elements(texp) gt 1) then flux = fltarr(n_elements(texp))
           if (n_elements(sn) gt 1) then flux = fltarr(n_elements(sn))
           if ~Keyword_set(quiet) then print, 'Compute Flux'
         end
else: begin
        print, 'ERROR: Unknown Action (',Action,') should be S, T or F'
        return
      end
endcase


; convert string character
Source = strupcase(strmid(Source, 0, 2))
Atm = strupcase(strmid(Atm, 0, 2))
Mode = strupcase(strmid(Mode, 0, 3))

; check if Source parameter is OK
case Source of
    'PC': if ~Keyword_set(quiet) then print, 'Source is Point-like & Continuum'
    'EC': if ~Keyword_set(quiet) then print, 'Source is Extended & Continuum'
    'PL': if ~Keyword_set(quiet) then print, 'Source is Point-like & Emission Line'
    'EL': if ~Keyword_set(quiet) then print, 'Source is Extended & Emission Line'
else: begin
        print, 'Error in parameter Source: ', Source
        return
      end
endcase

; check if Atm parameter is OK
case Atm of
    'NP': if ~Keyword_set(quiet) then print, 'Operating conditions are No AO & Poor atmospheric conditions'
    'NG': if ~Keyword_set(quiet) then print, 'Operating conditions are No AO & Good atmospheric conditions'
    'AP': if ~Keyword_set(quiet) then print, 'Operating conditions are AO & Poor atmospheric conditions'
    'AG': if ~Keyword_set(quiet) then print, 'Operating conditions are AO & Good atmospheric conditions'
else: begin
        print, 'Error in parameter Atm: ', Atm
        return
      end
endcase

; check if Mode parameter is OK
case Mode of
    'WFM': if ~Keyword_set(quiet) then print, 'MUSE Wide-Field Mode'
    'NFM': if ~Keyword_set(quiet) then print, 'MUSE Narrow-Field Mode'
else: begin
        print, 'Error in parameter Mode: ', Mode
        return
      end
endcase

hc = 6.626075510e-34*299792458.0 ; planck cste * light speed
VLT = 48.5425 ; VLT primary useful aperture in m2

DS = 1.135e-10 ; size of a spectral pixel in m
RN = 4.0 ; readout noise in e-
DC = 3.0/3600.0 ; dark current in e-/sec

case Mode of
    'WFM': DA = 0.2/206265.0 ; size of a spatial pixel in radian (WFM)
    'NFM': DA = 0.025/206265.0 ; size of a spatial pixel in radian (NFM)
endcase


Disp = 1
if Keyword_set(quiet) then begin
    Disp = 0
endif else begin
    if (n_elements(wave)*n_elements(flux)*n_elements(sn)*n_elements(texp) gt 20) then begin
        Disp = 0
        print, 'more than 20 elements computed ... print supressed'
    endif
endelse


n = 0
for i = 0, n_elements(Wave)-1 do begin
    lbda = wave[i]

    ; check wavelength is OK
    if ((lbda lt 4650) or (lbda gt 9300)) then begin
        print, 'Error Wavelength ', lbda, ' is outside MUSE limits (4650-9300)'
        return
    endif

    l = lbda*1.e-10 ; wavelength in m
    lmu = l*1.e6 ; wavelength in microns

    ; compute polynomial approximation of Extinction at Paranal
    Extinct = 3.60758-17.7112*lmu+34.2017*lmu^2 $
             -29.9169*lmu^3+9.84304*lmu^4
    Extinct = 10.0^(-0.4*Extinct*Airmass)

    ; compute polynomial approximation of MUSE+VLT total throughput
    case THROUGHPUT of
    ; typical WFM
       'T': Tmuse = -5.80097+27.3081*lmu-43.8719*lmu^2+30.2325*lmu^3-7.83402*lmu^4
    ; best WFM
       'B': Tmuse = -31.6506+298.614*lmu-1184.15*lmu^2+ 2500.34*lmu^3-2930.50*lmu^4+ $
                     1799.41*lmu^5-451.955*lmu^6
    ; case worst WFM
       'W': Tmuse = -20.6566+189.010*lmu-731.687*lmu^2+1519.74*lmu^3-1762.03*lmu^4+ $
                     1072.82*lmu^5-267.210*lmu^6
    endcase

    ; compute polynomial approximation of no OH Paranal Sky (in erg/s/cm2/arcsec2)
    FSky = (1351.28-13856.4*lmu+59775.8*lmu^2-140508.0*lmu^3+194373.0*lmu^4- $
           158272.0*lmu^5+70251.3*lmu^6-13113.5*lmu^7)*1.e-17
    FSky = FSky*206265.0^2*1.e7; convert in SI

    case Action of
       'S': begin
               for j = 0, n_elements(flux)-1 do begin
                 for k = 0, n_elements(texp)-1 do begin
                    Comp_SN, snc, flux[j], texp[k], Nexp, Source, Atm, Airmass, Mode, $
                        nspec, nspat
                    sn[n] = snc
                    n++
                 endfor
                endfor
            end
       'T': begin
               for j = 0, n_elements(flux)-1 do begin
                 for k = 0, n_elements(sn)-1 do begin
                    Comp_T, sn[k], flux[j], texpc, Nexp, Source, Atm, Airmass, Mode, $
                        nspec, nspat
                    texp[n] = texpc
                    n++
                 endfor
                endfor
            end
        'F': begin
               for j = 0, n_elements(texp)-1 do begin
                 for k = 0, n_elements(sn)-1 do begin
                    Comp_F, sn[k], fluxc, texp[j], Nexp, Source, Atm, Airmass, Mode, $
                        nspec, nspat
                    flux[n] = fluxc
                    n++
                 endfor
                endfor
            end
    endcase

endfor


if n gt 1 and ~Keyword_set(quiet) then begin
; we plot the results
    case Action of
       'S': begin
               if (n_elements(wave) gt 1) then begin
                 plot,wave,sn,xtitle='Wavelength',ytitle='S/N',$
                  title='S/N at Flux: '+ string(flux) + ' & ' + 'Integ/exp. ' + $
                   string(texp) + 's [ ' + string (nexp) + ' exp]'
               endif
               if (n_elements(flux) gt 1) then begin
                 plot,flux,sn,xtitle='Flux',ytitle='S/N',$
                  title='S/N at '+ string(wave) + 'A & Integ/exp. ' + $
                  string(texp) + 's [ ' + string (nexp) + ' exp]'
               endif
               if (n_elements(texp) gt 1) then begin
                 plot,texp,sn,xtitle='Integ/exp. (sec) [' + string(nexp) + $
                    ' exp]',ytitle='S/N',$
                  title='S/N at '+ string(wave) + 'A & Flux ' + string(flux)
               endif
            end
       'F': begin
               if (n_elements(wave) gt 1) then begin
                 plot,wave,flux,xtitle='Wavelength',ytitle='Flux',$
                  title='Flux for S/N '+ string(sn) + ' & ' + 'Integ/exp. ' + $
                   string(texp) + 's [ ' + string (nexp) + ' exp]'
               endif
               if (n_elements(sn) gt 1) then begin
                 plot,sn,flux,xtitle='S/N',ytitle='Flux',$
                  title='Flux at '+ string(wave) + 'A & Integ/exp. ' + $
                  string(texp) + 's [ ' + string (nexp) + ' exp]'
               endif
               if (n_elements(texp) gt 1) then begin
                 plot,texp,flux,xtitle='Integ/exp. (sec) [' + string(nexp) + $
                    ' exp]',ytitle='Flux',$
                  title='Flux at '+ string(wave) + 'A & S/N ' + string(sn)
               endif
            end
       'T': begin
               if (n_elements(wave) gt 1) then begin
                 plot,wave,texp,xtitle='Wavelength',ytitle='Integ/exp. ' + $
                    's [' + string(nexp) + ']', $
                    title='Int. Time/exp for S/N '+ string(sn) + ' & Flux ' + string (flux)
               endif
               if (n_elements(sn) gt 1) then begin
                 plot,sn,texp,xtitle='S/N',ytitle='Integ/exp. ' + $
                    's [' + string(nexp) + ']', $
                  title='Int. Time/exp at '+ string(wave) + 'A & Flux ' + string(flux)
               endif
               if (n_elements(flux) gt 1) then begin
                 plot,texp,texp,xtitle='Flux ' + string(flux), ytitle='Integ/exp. ' + $
                    's [' + string(nexp) + ']', $
                  title='Int. Time/exp at '+ string(wave) + 'A & S/N ' + string(sn)
               endif
            end
    endcase
endif


end


PRO FracSpa, Mode, Atm, lmu, FA, NA

; compute fraction of total flux enclosed in a spatial bin
case Mode of
    'NFM': begin
          ; case ao good
          FA = -0.893902+3.89988*lmu-3.99567*lmu^2+1.61919*lmu^3
          NA = 3*3 ; number of spatial pixel coadd in a spatial bin
         end

    'WFM': begin
         case Atm of
         'NP': begin ; case noao poor
                 FA = 0.276307+0.175628*lmu ;
                 NA = 4*4 ; number of spatial pixel coadd in a spatial bin
               end
         'NG': begin ; case noao good
                 FA = 0.335122+0.196059*lmu
                 NA = 3*3 ; number of spatial pixel coadd in a spatial bin
               end
         'AP': begin ; case ao poor
                 FA = 0.359636+0.288101*lmu
                 NA = 4*4 ; number of spatial pixel coadd in a spatial bin
               end
         'AG': begin ; case ao good
                 FA = 0.312652+0.615525*lmu-0.233294*lmu^2
                 NA = 3*3 ; number of spatial pixel coadd in a spatial bin
               end
         endcase
         end
endcase
end

PRO OnlyOne, a, b, c
if (n_elements(a) gt 1 and (n_elements(b) gt 1 or n_elements(c) gt 1) or $
    n_elements(b) gt 1 and (n_elements(a) gt 1 or n_elements(c) gt 1) or $
    n_elements(c) gt 1 and (n_elements(a) gt 1 or n_elements(b) gt 1)) then begin
    print, 'ERROR: Only one array permitted among sn,flux,texp,wave'
    return
endif
end

PRO Comp_SN, sn, flux, texp, Nexp, Source, Atm, Airmass, Mode, nspec, nspat
; computation of S/N given flux and integration time
Common Share

if (Disp eq 1) then print,'------------------------------------------------------------------------'

case Source of
    'PC': begin ; case Ponctual & Continuum
           if flag_AB eq 1 then begin
               FObj = AB2Flux(flux, lbda, /SI) ; convert AB in flux by A in SI unit
           endif else begin
               FObj = flux*1e7 ; convert flux in SI unit
           endelse
           FracSpa, Mode, Atm, lmu, FA, NA
           KO = double(nspec*DS*FA*Tmuse*VLT*Extinct*l/hc) ;
           KS = double(nspec*NA*DS*DA^2*Tmuse*VLT*l/hc) ;
           sn= sqrt(Nexp)*KO*FObj*texp/sqrt(KO*FObj*texp + KS*FSky*texp + nspec*NA*(RN^2 + DC*texp))
           if (Disp eq 1) then begin
              print,format='("Summed spaxels: ",i2, " fraction of object flux ",f4.2)', NA, FA
              if (nspec gt 1) then begin
                resolv = fix(l/(nspec*DS))
                print,format='("Low R: ", i4," : ", i3," spectral pixels sum up")', resolv, nspec
              endif
              if flag_AB eq 1 then begin
                  print,format='("Lbda: ",f4.2," m Object AB magnitude: ",f5.2)', lmu, flux
              endif else begin
                   print,format='("Lbda: ",f4.2," m Object flux: ",e10.2," erg/s/cm2/A")',$
                       lmu, flux
              endelse
              print_noise, KO*FObj*texp, KS*FSky*texp, nspec*NA*RN^2, nspec*NA*DC*texp
           endif
          end

    'PL': begin ; case Ponctual & Line
           if flag_AB eq 1 then begin
               FObj = AB2Flux(flux, lbda, /SI)*1e-10 ; convert AB in flux in SI unit
           endif else begin
               FObj = flux*1e-3 ; convert flux in SI unit
           endelse
           FracSpa, Mode, Atm, lmu, FA, NA
           NS = 3 ; number of spectral pixels coadd in the emission line
           FS = 0.923 ; fraction of line flux recovered when summing 3 pixels
           KO = double(FS*FA*Tmuse*VLT*Extinct*l/hc) ;
           KS = double(NS*NA*DS*DA^2*Tmuse*VLT*l/hc) ;
           sn = sqrt(Nexp)*KO*FObj*Texp/sqrt(KO*FObj*texp + KS*FSky*texp + NS*NA*(RN^2 + DC*texp))
           if (Disp eq 1) then begin
                print,format='("Summed spectral pixels: ",i2, " fraction of line flux ",f4.2)', $
                       NS, FS
                print,format='("Summed spaxels: ",i2, " fraction of object flux ",f4.2)', NA, FA
                if flag_AB eq 1 then begin
                    print,format='("Lbda: ",f4.2," m Object AB magnitude: ",f5.2)', lmu, flux
                endif else begin
                    print,format='("Lbda: ",f4.2," m Object flux: ",e10.2," erg/s/cm2")',$
                       lmu, flux
                endelse
                print_noise, KO*FObj*texp, KS*FSky*texp, NS*NA*RN^2, NS*NA*DC*texp
            endif
          end

    'EC': begin ; case Extended & Continuum
            if flag_AB eq 1 then begin
               FObj = AB2Flux(flux, lbda, /Arcsec, /SI) ; convert AB in flux by A and by arcsec2 in SI unit
            endif else begin
               FObj = flux*206265.0^2*1e7 ; convert flux in SI unit
            endelse
            KO = double(nspec*DS*nspat*DA^2*Tmuse*VLT*Extinct*l/hc) ;
            KS = double(nspec*DS*nspat*DA^2*Tmuse*VLT*l/hc) ;
            sn = sqrt(Nexp)*KO*FObj*Texp/sqrt(KO*FObj*texp + KS*FSky*texp + $
                   nspat*nspec*(RN^2 + DC*texp))
            if (Disp eq 1) then begin
                if (nspec gt 1) then begin
                    resolv = fix(l/(nspec*DS))
                    print,format='("Low R: ", i4," : ", i3," spectral pixels sum up")', resolv, nspec
                endif
                if (nspat gt 1) then begin
                   print,format='("Extra summed spatial pixels: ",i3, " - ",g10.4,"arcsec**2")',$
                     nspat, nspat*(DA*206265)^2
                endif
                if flag_AB eq 1 then begin
                    print,format='("Lbda: ",f4.2," m Object AB magnitude: ",f5.2," arcsec-2")', $
                        lmu, flux
                endif else begin
                    print,format='("Lbda: ",f4.2," m Object flux: ",e10.2," erg/s/cm2/A/arcsec2")', $
                        lmu, flux
                endelse
                print_noise, KO*FObj*texp, KS*FSky*texp, nspat*nspec*RN^2, nspat*nspec*DC*texp
            endif
          end

    'EL': begin ; case Extended & Line
            if flag_AB eq 1 then begin
               FObj = AB2Flux(flux, lbda, /Arcsec, /SI)*1e-10 ; convert AB in flux in SI unit
           endif else begin
               FObj = flux*206265.0^2*1e-3 ; convert flux in SI unit
           endelse
           NS = 3 ; number of spectral pixels coadd in the emission line
           FS = 0.923 ; fraction of line flux recovered when summing 3 pixels
           KO = double(FS*nspat*DA^2*Tmuse*VLT*Extinct*l/hc) ;
           KS = double(NS*DS*nspat*DA^2*Tmuse*VLT*l/hc) ;
           sn = sqrt(Nexp)*KO*FObj*Texp/sqrt(KO*FObj*texp + KS*FSky*texp + $
                nspat*NS*(RN^2 + DC*texp))
           if (Disp eq 1) then begin
                if (nspat gt 1) then begin
                   print,format='("Extra summed spatial pixels: ",i3, " - ",g10.4,"arcsec**2")',$
                     nspat, nspat*(DA*206265)^2
                endif
                print,format='("Summed spectral pixels: ",i2, " fraction of line flux ",f4.2)', $
                       NS, FS
                if Keyword_set(AB) then begin
                    print,format='("Lbda: ",f4.2," m Object AB magnitude: ",f5.2," arcsec-2")', $
                           lmu, flux
                endif else begin
                    print,format='("Lbda: ",f4.2," m Object flux: ",e10.2," erg/s/cm2/arcsec2")', $
                           lmu, flux
                endelse
                print_noise, KO*FObj*texp, KS*FSky*texp, nspat*NS*RN^2, nspat*NS*DC*texp
            endif
         end

endcase


if (Disp eq 1) then print,format='("S/N: ",f8.3, " in ",i2, " exposure(s) of ", f6.0," sec")', $
                  sn, Nexp, texp


return
end


PRO Comp_T, sn, flux, texp, Nexp, Source, Atm, Airmass, Mode, nspec, nspat
; compute of integration time given flux and S/N
Common Share

if (Disp eq 1) then print,'------------------------------------------------------------------------'

case Source of
    'PC': begin ; case Ponctual & Continuum
           if flag_AB eq 1 then begin
               FObj = AB2Flux(flux, lbda, /SI) ; convert AB in flux by A in SI unit
           endif else begin
               FObj = flux*1e7 ; convert flux in SI unit
           endelse
           FracSpa, Mode, Atm, lmu, FA, NA
           KO = double(nspec*DS*FA*Tmuse*VLT*Extinct*l/hc) ;
           KS = double(nspec*NA*DS*DA^2*Tmuse*VLT*l/hc) ;
           a = Nexp*(KO*Fobj/sn)^2
           b = KO*FObj + KS*FSky + nspec*NA*DC
           c = nspec*NA*RN^2
           texp = (b + sqrt(b^2+4*a*c))/(2*a)
           if (Disp eq 1) then begin
              if (nspec gt 1) then begin
                  resolv = fix(l/(nspec*DS))
                  print,format='("Low R: ", i4," : ", i3," spectral pixels sum up")', resolv, nspec
              endif
              print,format='("Summed spaxels: ",i2, " fraction of object flux ",f4.2)', NA, FA
              if flag_AB eq 1 then begin
                  print,format='("Lbda: ",f4.2," m Object AB magnitude: ",f5.2)', lmu, flux
              endif else begin
                   print,format='("Lbda: ",f4.2," m Object flux: ",e10.2," erg/s/cm2/A")',$
                       lmu, flux
              endelse
              print_noise, KO*FObj*texp, KS*FSky*texp, nspec*NA*RN^2, nspec*NA*DC*texp
            endif
          end

    'PL': begin ; case Ponctual & Line
           if flag_AB eq 1 then begin
               FObj = AB2Flux(flux, lbda, /SI)*1e-10 ; convert AB in flux in SI unit
           endif else begin
               FObj = flux*1e-3 ; convert flux in SI unit
           endelse
           FracSpa, Mode, Atm, lmu, FA, NA
           NS = 3 ; number of spectral pixels coadd in the emission line
           FS = 0.923 ; fraction of line flux recovered when summing 3 pixels
           KO = double(FS*FA*Tmuse*VLT*Extinct*l/hc) ;
           KS = double(NS*NA*DS*DA^2*Tmuse*VLT*l/hc) ;
           a = Nexp*(KO*Fobj/sn)^2
           b = KO*FObj + KS*FSky + NS*NA*DC
           c = NS*NA*RN^2
           texp = (b + sqrt(b^2+4*a*c))/(2*a)
           if (Disp eq 1) then begin
                print,format='("Summed spectral pixels: ",i2, " fraction of line flux ",f4.2)', $
                       NS, FS
                print,format='("Summed spaxels: ",i2, " fraction of object flux ",f4.2)', NA, FA
                if flag_AB eq 1 then begin
                    print,format='("Lbda: ",f4.2," m Object AB magnitude: ",f5.2)', lmu, flux
                endif else begin
                    print,format='("Lbda: ",f4.2," m Object flux: ",e10.2," erg/s/cm2")',$
                       lmu, flux
                endelse
                print_noise, KO*FObj*texp, KS*FSky*texp, NS*NA*RN^2, NS*NA*DC*texp
            endif
          end

    'EC': begin ; case Extended & Continuum
            if flag_AB eq 1 then begin
               FObj = AB2Flux(flux, lbda, /Arcsec, /SI) ; convert AB in flux by A and by arcsec2 in SI unit
            endif else begin
               FObj = flux*206265.0^2*1e7 ; convert flux in SI unit
            endelse
            KO = double(nspec*DS*nspat*DA^2*Tmuse*VLT*Extinct*l/hc) ;
            KS = double(nspec*DS*nspat*DA^2*Tmuse*VLT*l/hc) ;
            a = Nexp*(KO*Fobj/sn)^2
            b = KO*FObj + KS*FSky + nspat*nspec*DC
            c = nspat*nspec*RN^2
            texp = (b + sqrt(b^2+4*a*c))/(2*a)
            if (Disp eq 1) then begin
                if (nspec gt 1) then begin
                    resolv = fix(l/(nspec*DS))
                    print,format='("Low R: ", i4," : ", i3," spectral pixels sum up")', resolv, nspec
                endif
                if (nspat gt 1) then begin
                   print,format='("Extra summed spatial pixels: ",i3, " - ",g10.4,"arcsec**2")',$
                     nspat, nspat*(DA*206265)^2
                endif
                if flag_AB eq 1 then begin
                    print,format='("Lbda: ",f4.2," m Object AB magnitude: ",f5.2," arcsec-2")', $
                        lmu, flux
                endif else begin
                    print,format='("Lbda: ",f4.2," m Object flux: ",e10.2," erg/s/cm2/A/arcsec2")', $
                        lmu, flux
                endelse
                print_noise, KO*FObj*texp, KS*FSky*texp, nspat*nspec*RN^2, nspat*nspec*DC*texp
            endif
          end

    'EL': begin ; case Extended & Line
            if flag_AB eq 1 then begin
               FObj = AB2Flux(flux, lbda, /Arcsec, /SI)*1e-10 ; convert AB in flux in SI unit
           endif else begin
               FObj = flux*206265.0^2*1e-3 ; convert flux in SI unit
           endelse
           NS = 3 ; number of spectral pixels coadd in the emission line
           FS = 0.923 ; fraction of line flux recovered when summing 3 pixels
           KO = double(FS*nspat*DA^2*Tmuse*VLT*Extinct*l/hc) ;
           KS = double(NS*DS*nspat*DA^2*Tmuse*VLT*l/hc) ;
           a = Nexp*(KO*Fobj/sn)^2
           b = KO*FObj + KS*FSky + nspat*NS*DC
           c = nspat*NS*RN^2
           texp = (b + sqrt(b^2+4*a*c))/(2*a)
           if (Disp eq 1) then begin
                if (nspat gt 1) then begin
                   print,format='("Extra summed spatial pixels: ",i3, " - ",g10.4,"arcsec**2")',$
                     nspat, nspat*(DA*206265)^2
                endif
                print,format='("Summed spectral pixels: ",i2, " fraction of line flux ",f4.2)', $
                       NS, FS
                if Keyword_set(AB) then begin
                    print,format='("Lbda: ",f4.2," m Object AB magnitude: ",f5.2," arcsec-2")', $
                           lmu, flux
                endif else begin
                    print,format='("Lbda: ",f4.2," m Object flux: ",e10.2," erg/s/cm2/arcsec2")', $
                           lmu, flux
                endelse
                print_noise, KO*FObj*texp, KS*FSky*texp, nspat*NS*RN^2, nspat*NS*DC*texp
            endif
         end

endcase


if (Disp eq 1) then print,format='("S/N: ",f8.3, " in ",i2, " exposure(s) of ", f6.0," sec")', $
                  sn, Nexp, texp

return
end

PRO Comp_F, sn, flux, texp, Nexp, Source, Atm, Airmass, Mode, nspec, nspat
; compute flux given integration time and S/N
Common Share

if (Disp eq 1) then print,'------------------------------------------------------------------------'

case Source of
    'PC': begin ; case Ponctual & Continuum
           FracSpa, Mode, Atm, lmu, FA, NA
           KO = double(nspec*DS*FA*Tmuse*VLT*Extinct*l/hc) ;
           KS = double(nspec*NA*DS*DA^2*Tmuse*VLT*l/hc) ;
           a = Nexp*(KO*texp/sn)^2
           b = KO*texp
           c = KS*FSky*texp + nspec*NA*DC*texp + nspec*NA*RN^2
           Fobj = (b + sqrt(b^2+4*a*c))/(2*a)
           if flag_AB eq 1 then begin
               flux = Flux2AB(Fobj, lbda, /SI) ; convert flux by A in SI unit in AB mag
           endif else begin
               flux = Fobj*1e-7 ; convert flux in erg/s/cm-2/A
           endelse
           if (Disp eq 1) then begin
              if (nspec gt 1) then begin
                  resolv = fix(l/(nspec*DS))
                  print,format='("Low R: ", i4," : ", i3," spectral pixels sum up")', resolv, nspec
              endif
              print,format='("Summed spaxels: ",i2, " fraction of object flux ",f4.2)', NA, FA
              if flag_AB eq 1 then begin
                  print,format='("Lbda: ",f4.2," m Object AB magnitude: ",f5.2)', lmu, flux
              endif else begin
                   print,format='("Lbda: ",f4.2," m Object flux: ",e10.2," erg/s/cm2/A")',$
                       lmu, flux
              endelse
              print_noise, KO*FObj*texp, KS*FSky*texp, nspec*NA*RN^2, nspec*NA*DC*texp
           endif
          end

    'PL': begin ; case Ponctual & Line
           FracSpa, Mode, Atm, lmu, FA, NA
           NS = 3 ; number of spectral pixels coadd in the emission line
           FS = 0.923 ; fraction of line flux recovered when summing 3 pixels
           KO = double(FS*FA*Tmuse*VLT*Extinct*l/hc) ;
           KS = double(NS*NA*DS*DA^2*Tmuse*VLT*l/hc) ;
           a = Nexp*(KO*texp/sn)^2
           b = KO*texp
           c = KS*FSky*texp + NA*NS*DC*texp + NA*NS*RN^2
           Fobj = (b + sqrt(b^2+4*a*c))/(2*a)
           if flag_AB eq 1 then begin
               flux = Flux2AB(Fobj, lbda, /SI) ; convert flux by A in SI unit in AB mag
           endif else begin
               flux = Fobj*1e3 ; convert flux in erg/s/cm-2
           endelse
           if (Disp eq 1) then begin
                print,format='("Summed spectral pixels: ",i2, " fraction of line flux ",f4.2)', $
                       NS, FS
                print,format='("Summed spaxels: ",i2, " fraction of object flux ",f4.2)', NA, FA
                if flag_AB eq 1 then begin
                    print,format='("Lbda: ",f4.2," m Object AB magnitude: ",f5.2)', lmu, flux
                endif else begin
                    print,format='("Lbda: ",f4.2," m Object flux: ",e10.2," erg/s/cm2")',$
                       lmu, flux
                endelse
                print_noise, KO*FObj*texp, KS*FSky*texp, NS*NA*RN^2, NS*NA*DC*texp
            endif
          end

    'EC': begin ; case Extended & Continuum
            KO = double(nspec*DS*nspat*DA^2*Tmuse*VLT*Extinct*l/hc) ;
            KS = double(nspec*DS*nspat*DA^2*Tmuse*VLT*l/hc) ;
            a = Nexp*(KO*texp/sn)^2
            b = KO*texp
            c = KS*FSky*texp + nspat*nspec*DC*texp + nspat*nspec*RN^2
            Fobj = (b + sqrt(b^2+4*a*c))/(2*a)
            if flag_AB eq 1 then begin
               flux = Flux2AB(Fobj, lbda, /SI, /arcsec) ; convert flux by A in SI unit in AB mag
            endif else begin
               flux = Fobj*1e-7/206265.0^2 ; convert flux in erg/s/cm2/A/arcsec2
            endelse
            if (Disp eq 1) then begin
                if (nspec gt 1) then begin
                  resolv = fix(l/(nspec*DS))
                  print,format='("Low R: ", i4," : ", i3," spectral pixels sum up")', resolv, nspec
                endif
                if (nspat gt 1) then begin
                   print,format='("Extra summed spatial pixels: ",i3, " - ",g10.4,"arcsec**2")',$
                     nspat, nspat*(DA*206265)^2
                endif

                if flag_AB eq 1 then begin
                    print,format='("Lbda: ",f4.2," m Object AB magnitude: ",f5.2," arcsec-2")', $
                        lmu, flux
                endif else begin
                    print,format='("Lbda: ",f4.2," m Object flux: ",e10.2," erg/s/cm2/A/arcsec2")', $
                        lmu, flux
                endelse
                print_noise, KO*FObj*texp, KS*FSky*texp, nspat*nspec*RN^2, nspat*nspec*DC*texp
            endif
          end

    'EL': begin ; case Extended & Line
           NS = 3 ; number of spectral pixels coadd in the emission line
           FS = 0.923 ; fraction of line flux recovered when summing 3 pixels
           KO = double(FS*nspat*DA^2*Tmuse*VLT*Extinct*l/hc) ;
           KS = double(NS*DS*nspat*DA^2*Tmuse*VLT*l/hc) ;
           a = Nexp*(KO*texp/sn)^2
           b = KO*texp
           c = KS*FSky*texp + nspat*NS*DC*texp + nspat*NS*RN^2
           Fobj = (b + sqrt(b^2+4*a*c))/(2*a)
           if flag_AB eq 1 then begin
              flux = Flux2AB(Fobj, lbda, /SI, /arcsec) ; convert flux by A in SI unit in AB mag
           endif else begin
              flux = Fobj*1e3/206265.0^2 ; convert flux in erg/s/cm2/arcsec2
           endelse
           if (Disp eq 1) then begin
                if (nspat gt 1) then begin
                   print,format='("Extra summed spatial pixels: ",i3, " - ",g10.4,"arcsec**2")',$
                     nspat, nspat*(DA*206265)^2
                endif
                print,format='("Summed spectral pixels: ",i2, " fraction of line flux ",f4.2)', $
                       NS, FS
                if Keyword_set(AB) then begin
                    print,format='("Lbda: ",f4.2," m Object AB magnitude: ",f5.2," arcsec-2")', $
                           lmu, flux
                endif else begin
                    print,format='("Lbda: ",f4.2," m Object flux: ",e10.2," erg/s/cm2/arcsec2")', $
                           lmu, flux
                endelse
                print_noise, KO*FObj*texp, KS*FSky*texp, nspat*NS*RN^2, nspat*NS*DC*texp
            endif
         end

endcase


if (Disp eq 1) then print,format='("S/N: ",f8.3, " in ",i2, " exposure(s) of ", f6.0," sec")', $
                  sn, Nexp, texp

return
end

PRO print_noise, VObj, VSky, VRN, VDC
; compute and print statistics in noise variance
Vtot = VObj + VSky + VRN + VDC

print, format='("Noise variance statistics - Obj:",f5.1,"% Sky:",f5.1,"% Detector:",f5.1,"% [RN:",f5.1, "% DC:",f5.1,"]")', $
    100*VObj/Vtot, 100*VSky/Vtot, 100*(VRN+VDC)/Vtot, 100*VRN/Vtot, 100*VDC/Vtot
end
