/* $Id: vircam_mkconf.c,v 1.11 2007/03/29 12:19:39 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2005 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/03/29 12:19:39 $
 * $Revision: 1.11 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "vircam_mods.h"
#include "vircam_utils.h"
#include "vircam_mask.h"
#include "vircam_stats.h"
#include "vircam_fits.h"

/**@{*/

/*---------------------------------------------------------------------------*/
/**
    \ingroup reductionmodules
    \brief Create a confidence map

    \par Name:
        vircam_mkconf
    \par Purpose:
        Create a confidence map
    \par Description:
        Create a confidence map from a given flat image and a bad pixel mask.
	The confidence is defined as 100 times the normalised flat field. Any
	pixels that are flagged as bad will have a value of 0 confidence.
    \par Language:
        C
    \param flat
        The input flat field data image 
    \param flatfile
        The input flat field file name.
    \param bpm
        The input mask image
    \param outconf
        The ouput confidence map image
    \param drs
        A propertylist with entries for the drs header block
    \param status
        An input/output status that is the same as the returned values below.
    \retval VIR_OK
        if everything is ok
    \retval VIR_FATAL
        if image data fails to load or the dimensionality of the input images
	don't match.
    \par QC headers:
        None
    \par DRS headers:
        The following DRS keywords are written to the drs propertylist
	- \b FLATIN
	    The name of the originating FITS file for the flat image
        - \b BPMIN: 
	    The name of the originating FITS file for the BPM image
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_mkconf(cpl_image *flat, char *flatfile, vir_mask *bpm, 
			 cpl_image **outconf, cpl_propertylist **drs, 
			 int *status) {
    int i,*odata,nx,ny;
    long npts;
    unsigned char *bdata;
    float *fdata,mean;
    const char *fctid = "vircam_mkconf";

    /* Inherited status */

    if (*status != VIR_OK)
        return(*status);

    /* Check the sizes of the input images to make sure they match */

    nx = cpl_image_get_size_x(flat);
    ny = cpl_image_get_size_y(flat);
    npts = nx*ny;
    if (vircam_mask_get_size_x(bpm)*vircam_mask_get_size_y(bpm) != npts) {
        cpl_msg_error(fctid,"Input image sizes don't match!");
   	FATAL_ERROR
    }

    /* Get input the data arrays */

    if ((fdata = cpl_image_get_data_float(flat)) == NULL) {
	cpl_msg_error(fctid,"Unable to map flat data!");
	FATAL_ERROR
    }
    bdata = vircam_mask_get_data(bpm);

    /* Get a data array for the output */

    odata = cpl_malloc(npts*sizeof(*odata));

    /* Work out the mean of the flat field. It should already be 1, but you
       never know... */

    mean = vircam_mean(fdata,bdata,npts);

    /* Now create the output array */

    for (i = 0; i < npts; i++) {
	if (bdata[i] != 1)
	    odata[i] = min(110,(int)(100.0*fdata[i]/mean + 0.5));
	else
	    odata[i] = 0;
    }

    /* Wrap the data into an output image */

    *outconf = cpl_image_wrap_int(nx,ny,odata);

    /* Create some properties */

    *drs = cpl_propertylist_new();
    cpl_propertylist_update_string(*drs,"ESO DRS FLATIN",flatfile);
    cpl_propertylist_set_comment(*drs,"ESO DRS FLATIN",
				 "Flat used to create this conf map");
    if (vircam_mask_get_type(bpm) != MASK_NONE) 
        cpl_propertylist_update_string(*drs,"ESO DRS BPMIN",
				       vircam_fits_get_fullname(vircam_mask_get_fits(bpm)));
    else
        cpl_propertylist_update_string(*drs,"ESO DRS BPMIN","None available");
    cpl_propertylist_set_comment(*drs,"ESO DRS BPMIN",
				 "BPM used to create this conf map");

    /* Tidy up */

    GOOD_STATUS
}

/**@}*/

/*

$Log: vircam_mkconf.c,v $
Revision 1.11  2007/03/29 12:19:39  jim
Little changes to improve documentation

Revision 1.10  2007/03/01 12:42:42  jim
Modified slightly after code checking

Revision 1.9  2006/04/20 11:27:28  jim
Error messages now print an extension name rather than an extension number

Revision 1.8  2006/03/23 21:18:52  jim
Minor changes mainly to comment headers

Revision 1.7  2006/03/22 13:58:32  jim
Cosmetic fixes to keep lint happy

Revision 1.6  2006/03/22 13:19:19  jim
Altered calling sequence slightly

Revision 1.5  2006/03/22 12:02:40  jim
bpm is accessed using vircam_mask interface

Revision 1.4  2006/03/15 10:43:41  jim
Fixed a few things

Revision 1.3  2006/03/01 10:31:29  jim
Now uses new vir_fits objects

Revision 1.2  2006/02/22 14:08:07  jim
Fixed silly doc error

Revision 1.1  2006/02/18 11:46:36  jim
new file


*/
