/* $Id: vircam_matchxy.c,v 1.11 2007/10/25 19:38:22 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2006 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/10/25 19:38:22 $
 * $Revision: 1.11 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <cpl.h>

#include "vircam_utils.h"
#include "vircam_dfs.h"
#include "vircam_fits.h"
#include "vircam_mods.h"
#include "vircam_paf.h"

/* Function prototypes */

static int vircam_matchxy_create(cpl_plugin *);
static int vircam_matchxy_exec(cpl_plugin *);
static int vircam_matchxy_destroy(cpl_plugin *);
static int vircam_matchxy_test(cpl_parameterlist *, cpl_frameset *);
static int vircam_matchxy_save(void);
static void vircam_matchxy_init(void);
static void vircam_matchxy_tidy(void);

static struct {

    /* Input */

    int         extenum;

    /* Output */

    float       xoff;
    float       yoff;
    int         nm;

} vircam_matchxy_config;


static struct {
    int          *labels;
    cpl_frameset *catlist;
    cpl_frame    *cat1;
    cpl_frame    *cat2;
    vir_tfits    *cat1f;
    vir_tfits    *cat2f;
} ps;


static char vircam_matchxy_description[] =
"vircam_matchxy -- VIRCAM recipe to test vircam_matchxy.\n\n"
"Match a catalogue with another to get x,y offsets\n\n"
"The program accepts the following files in the SOF:\n\n"
"    Tag                   Description\n"
"    -----------------------------------------------------------------------\n"
"    %-21s Input catalogues of objects extracted from an image\n"
"\n";

/**@{*/

/**
    \ingroup testrecipelist
    \defgroup vircam_matchxy vircam_matchxy
    \brief Test recipe to drive the vircam_matchxy library function.

    \par Name:
        vircam_matchxy
    \par Purpose:
        Test recipe to drive the vircam_matchxy library function.
    \par Description:
        Test the library function vircam_matchxy by matching an object
	catalogue with another to work out the x,y offset.
    \par Language:
        C
    \par Parameters:
        - \b ext (int): The image extension of the input files to be done
          on this run. If all of the extensions are to be processed, then
          this should be set to zero.
    \par Input File Types:
        The following list gives the file types that can appear in the SOF
        file. The word in bold is the DO category value.
        - \b OBJECT_CATALOGUE (required): An object catalogue
    \par Output File Types:
        None. Results written to PAF.
    \par Notes
        None.
    \par Fatal Error Conditions:
        - Missing input tables
    \par Non-Fatal Error Conditions:
        None.
    \par Author:
        Jim Lewis, CASU
    \par Code Reference:
        tests/vircam_matchxy.c
*/

/* Function code */

/*---------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok

  This function is exported.
 */
/*---------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list) {
    cpl_recipe  *recipe = cpl_calloc(1,sizeof(*recipe));
    cpl_plugin  *plugin = &recipe->interface;
    char alldesc[SZ_ALLDESC];
    (void)snprintf(alldesc,SZ_ALLDESC,vircam_matchxy_description,
		   VIRCAM_CAL_OBJCAT);

    cpl_plugin_init(plugin,
                    CPL_PLUGIN_API,
                    VIRCAM_BINARY_VERSION,
                    CPL_PLUGIN_TYPE_RECIPE,
                    "vircam_matchxy",
                    "VIRCAM catalogue matching test recipe [test]",
                    alldesc,
                    "Jim Lewis",
                    "jrl@ast.cam.ac.uk",
                    vircam_get_license(),
                    vircam_matchxy_create,
                    vircam_matchxy_exec,
                    vircam_matchxy_destroy);

    cpl_pluginlist_append(list,plugin);

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*---------------------------------------------------------------------------*/

static int vircam_matchxy_create(cpl_plugin *plugin) {
    cpl_recipe      *recipe;
    cpl_parameter   *p;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    /* Create the parameters list in the cpl_recipe object */

    recipe->parameters = cpl_parameterlist_new();

    /* Extension number of input frames to use */

    p = cpl_parameter_new_range("vircam.vircam_matchxy.extenum",
                                CPL_TYPE_INT,
                                "Extension number to be done, 0 == all",
                                "vircam.vircam_matchxy",1,0,16);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"ext");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Get out of here */

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_matchxy_exec(cpl_plugin *plugin) {
    cpl_recipe  *recipe;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    return(vircam_matchxy_test(recipe->parameters,recipe->frames));
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_matchxy_destroy(cpl_plugin *plugin) {
    cpl_recipe *recipe ;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    cpl_parameterlist_delete(recipe->parameters);
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data reduction part is implemented here
  @param    parlist     the parameters list
  @param    framelist   the frames list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_matchxy_test(cpl_parameterlist *parlist, 
			       cpl_frameset *framelist) {
    const char *fctid="vircam_matchxy";
    cpl_parameter *p;
    int nlab,jst,jfn,status,j;

    /* Check validity of input frameset */

    if (framelist == NULL || cpl_frameset_get_size(framelist) <= 0) {
        cpl_msg_error(fctid,"Input framelist NULL or has no input data\n");
        return(-1);
    }

    /* Initialise some things */

    vircam_matchxy_init();

    /* Get the parameters */

    p = cpl_parameterlist_find(parlist,"vircam.vircam_matchxy.extenum");
    vircam_matchxy_config.extenum = cpl_parameter_get_int(p);

    /* Sort out raw from calib frames */

    if (vircam_dfs_set_groups(framelist) != VIR_OK) {
        cpl_msg_error(fctid,"Cannot identify RAW and CALIB frames");
        vircam_matchxy_tidy();
        return(-1);
    }

    /* Get the frames */

    if ((ps.labels = cpl_frameset_labelise(framelist,vircam_compare_tags,
                                           &nlab)) == NULL) {
        cpl_msg_error(fctid,"Cannot labelise the input frames");
        vircam_matchxy_tidy();
        return(-1);
    }
    if ((ps.catlist = vircam_frameset_subgroup(framelist,ps.labels,nlab,
					       VIRCAM_CAL_OBJCAT)) == NULL) {
        cpl_msg_error(fctid,"No object catalogues found -- cannot continue");
	vircam_matchxy_tidy();
	return(-1);
    }
    ps.cat1 = cpl_frameset_get_frame(ps.catlist,0);
    ps.cat2 = cpl_frameset_get_frame(ps.catlist,1);
    if (ps.cat1 == NULL || ps.cat2 == NULL) {
	cpl_msg_error(fctid,"List does not contain two object catalogues");
        vircam_matchxy_tidy();
	return(-1);
    }

    /* Now, how many image extensions do we want to do? If the extension
       number is zero, then we loop for all possible extensions. If it
       isn't then we just do the extension specified */

    vircam_exten_range(vircam_matchxy_config.extenum,
		       (const cpl_frame *)ps.cat1,&jst,&jfn);
    if (jst == -1 || jfn == -1) {
	cpl_msg_error(fctid,"Unable to continue");
	vircam_matchxy_tidy();
	return(-1);
    }

    /* Now loop for all the extension... */

    status = VIR_OK;
    for (j = jst; j <= jfn; j++) {

        /* Load up the tables */

        ps.cat1f = vircam_tfits_load(ps.cat1,j);
        ps.cat2f = vircam_tfits_load(ps.cat2,j);
        if (ps.cat1f == NULL || ps.cat2f == NULL) {
	    cpl_msg_warning(fctid,"No matching done\n");
	    continue;
	}

	/* Now do the correction */

	cpl_msg_info(fctid,"Doing the matching for extension %d",j);
        (void)vircam_matchxy(vircam_tfits_get_table(ps.cat1f),
			     vircam_tfits_get_table(ps.cat2f),200.0,
			     &(vircam_matchxy_config.xoff),
			     &(vircam_matchxy_config.yoff),
			     &(vircam_matchxy_config.nm),&status);
	if (status != VIR_OK) {
	    cpl_msg_warning(fctid,"No matching results done\n");
	    status = VIR_OK;
	}

        /* Now save the result */

	cpl_msg_info(fctid,"Saving results for extension %d",j);
	if (vircam_matchxy_save() != 0) 
	    cpl_msg_warning(fctid,"No matching results saved\n");

	/* Tidy a few things before the next image */

	freetfits(ps.cat1f);
	freetfits(ps.cat2f);
    }
    vircam_matchxy_tidy();
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data products are saved here
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_matchxy_save(void) {
    const char *fctid = "vircam_matchxy_save";
    const char *outfile = "matchxy";
    cpl_propertylist *p,*p2,*p3;

    /* Get a propertylist from the first file extension */

    p = vircam_tfits_get_ehu(ps.cat1f);

    /* Remove this line when they fix cpl so that the date-obs isn't removed
       from the table extensions */

    cpl_propertylist_update_string(p,"DATE-OBS","ABC");

    /* Extract the required things */

    p2 = vircam_paf_req_items(p);
    p3 = vircam_paf_phu_items(vircam_tfits_get_phu(ps.cat1f));
    vircam_merge_propertylists(p2,p3);
    freepropertylist(p3);

    /* Add some new stuff in */

    cpl_propertylist_update_float(p2,"ESO QC XOFF",vircam_matchxy_config.xoff);
    cpl_propertylist_set_comment(p2,"ESO QC XOFF",
				 "Calculated X offset (pixels)");
    cpl_propertylist_update_float(p2,"ESO QC YOFF",vircam_matchxy_config.yoff);
    cpl_propertylist_set_comment(p2,"ESO QC YOFF",
				 "Calculated Y offset (pixels)");
    cpl_propertylist_update_int(p2,"ESO QC NUMMATCH",
				  vircam_matchxy_config.nm);
    cpl_propertylist_set_comment(p2,"ESO QC NUMMATCH",
				 "Number of matching objects");

    /* Now write the PAF */

    if (vircam_paf_print((char *)outfile,"VIRCAM/vircam_matchxy",
			 "Test output file",p2) != VIR_OK) {
	cpl_msg_error(fctid,"Unable to write PAF");
	cpl_propertylist_delete(p2);
	return(-1);
    }
    cpl_propertylist_delete(p2);
    return(0);
}



/*---------------------------------------------------------------------------*/
/**
  @brief    Initialise the pointers in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_matchxy_init(void) {
    ps.labels = NULL;
    ps.cat1 = NULL;
    ps.cat2 = NULL;
    ps.cat1f = NULL;
    ps.cat2f = NULL;
    ps.catlist = NULL;
}


/*---------------------------------------------------------------------------*/
/**
  @brief    Free any allocated workspace in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_matchxy_tidy(void) {
    freespace(ps.labels);
    freetfits(ps.cat1f);
    freetfits(ps.cat2f);
    freeframeset(ps.catlist);
}

/**@}*/

/*

$Log: vircam_matchxy.c,v $
Revision 1.11  2007/10/25 19:38:22  jim
modified to keep lint happy

Revision 1.10  2007/07/09 13:22:09  jim
Modified to use new version of vircam_exten_range

Revision 1.9  2007/04/23 12:49:07  jim
Changed behaviour for error condition

Revision 1.8  2007/04/13 12:27:39  jim
Added some extra docs

Revision 1.7  2007/04/04 10:36:29  jim
Modified to use new dfs tags

Revision 1.6  2007/03/01 12:42:59  jim
Modified slightly after code checking

Revision 1.5  2007/02/15 12:17:45  jim
Modified to use new version of PAF files

Revision 1.4  2006/06/15 09:59:00  jim
Minor changes to docs

Revision 1.3  2006/05/04 11:53:44  jim
Fixed _save routine so that it's more consistent with the standard CPL
way of doing things

Revision 1.2  2006/04/27 14:22:05  jim
Fixed docs

Revision 1.1  2006/04/24 10:42:45  jim
New routine


*/




