/* $Id: vircam_lincor.c,v 1.11 2007/11/26 09:56:04 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2006 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/11/26 09:56:04 $
 * $Revision: 1.11 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <cpl.h>

#include "vircam_utils.h"
#include "vircam_dfs.h"
#include "vircam_fits.h"
#include "vircam_mods.h"

/* Function prototypes */

static int vircam_lincor_create(cpl_plugin *) ;
static int vircam_lincor_exec(cpl_plugin *) ;
static int vircam_lincor_destroy(cpl_plugin *) ;
static int vircam_lincor_test(cpl_parameterlist *, cpl_frameset *) ;
static int vircam_lincor_save(cpl_frameset *framelist, 
			      cpl_parameterlist *parlist);
static void vircam_lincor_init(void);
static void vircam_lincor_tidy(void);

static struct {

    /* Input */

    int         extenum;

} vircam_lincor_config;

static struct {
    int         *labels;
    cpl_frame   *chan;
    cpl_frame   *img;
    vir_tfits   *chanf;
    vir_fits    *imgf;
} ps;

static int isfirst;
static cpl_frame *product_frame = NULL;

static char vircam_lincor_description[] =
"vircam_lincor -- VIRCAM linearity correction test recipe.\n\n"
"Linearity correct an input frame using a pre-existing channel table\n\n"
"The program accepts the following files in the SOF:\n\n"
"    Tag                   Description\n"
"    -----------------------------------------------------------------------\n"
"    %-21s A input uncorrected image\n"
"    %-21s A channel table\n"
"\n";

/**@{*/

/**
    \ingroup testrecipelist
    \defgroup vircam_lincor vircam_lincor
    \brief Test recipe to drive the vircam_lincor library function.

    \par Name:
        vircam_lincor
    \par Purpose:
        Test recipe to drive the vircam_lincor library function.
    \par Description:
        Test the library function vircam_lincor by linearity correcting
        frame an input uncorrected frame. Although it is not necessary
	that the input frame should be dark corrected and flat fielded
        before using it in this routine, the results will be more satisfactory
	if it is.
    \par Language:
        C
    \par Parameters:
        - \b ext (int): The image extension of the input files to be done
          on this run. If all of the extensions are to be processed, then
          this should be set to zero.
    \par Input File Types:
        The following list gives the file types that can appear in the SOF
        file. The word in bold is the DO category value.
        - \b SCIENCE_IMAGE (required): An input (prefereably dark corrected) 
	  image.
        - \b MASTER_CHANNEL_TABLE (required): A master channel table
    \par Output File Types:
        The following lists the output data frames that are generated by 
        this recipe. The word in bold gives the PRO CATG keyword value for 
        each.
        - \b SIMPLE_IMAGE_TEST: A linearity corrected image.
    \par Notes
        None.
    \par Fatal Error Conditions:
        - Missing input image or channel table
    \par Non-Fatal Error Conditions:
        None.
    \par Author:
        Jim Lewis, CASU
    \par Code Reference:
        tests/vircam_lincor.c
*/

/* Function code */

/*---------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok

  This function is exported.
 */
/*---------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list) {
    cpl_recipe  *recipe = cpl_calloc(1,sizeof(*recipe));
    cpl_plugin  *plugin = &recipe->interface;
    char alldesc[SZ_ALLDESC];
    (void)snprintf(alldesc,SZ_ALLDESC,vircam_lincor_description,
                   VIRCAM_TEST_SCIENCE_RAW,VIRCAM_CAL_CHANTAB);

    cpl_plugin_init(plugin,
                    CPL_PLUGIN_API,
                    VIRCAM_BINARY_VERSION,
                    CPL_PLUGIN_TYPE_RECIPE,
                    "vircam_lincor",
                    "VIRCAM linearisation test recipe [test]",
                    alldesc,
                    "Jim Lewis",
                    "jrl@ast.cam.ac.uk",
                    vircam_get_license(),
                    vircam_lincor_create,
                    vircam_lincor_exec,
                    vircam_lincor_destroy);

    cpl_pluginlist_append(list,plugin);

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*---------------------------------------------------------------------------*/

static int vircam_lincor_create(cpl_plugin *plugin) {
    cpl_recipe      *recipe;
    cpl_parameter   *p;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    /* Create the parameters list in the cpl_recipe object */

    recipe->parameters = cpl_parameterlist_new();

    /* Get the extension number of input frames to use */

    p = cpl_parameter_new_range("vircam.vircam_lincor.extenum",
                                CPL_TYPE_INT,
                                "Extension number to be done, 0 == all",
                                "vircam.vircam_lincor",1,0,16);
    cpl_parameter_set_alias(p,CPL_PARAMETER_MODE_CLI,"ext");
    cpl_parameterlist_append(recipe->parameters,p);

    /* Get out of here */

    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_lincor_exec(cpl_plugin *plugin) {
    cpl_recipe  *recipe;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    return(vircam_lincor_test(recipe->parameters,recipe->frames));
}


/*---------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_lincor_destroy(cpl_plugin *plugin) {
    cpl_recipe *recipe ;

    /* Get the recipe out of the plugin */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE)
        recipe = (cpl_recipe *)plugin;
    else
        return(-1);

    cpl_parameterlist_delete(recipe->parameters);
    return(0);
}

/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data reduction part is implemented here
  @param    parlist     the parameters list
  @param    framelist   the frames list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_lincor_test(cpl_parameterlist *parlist, 
			      cpl_frameset *framelist) {
    const char *fctid="vircam_lincor";
    cpl_parameter *p;
    int nlab,jst,jfn,status,j,ndit;
    cpl_propertylist *pp;

    /* Check validity of input frameset */

    if (framelist == NULL || cpl_frameset_get_size(framelist) <= 0) {
        cpl_msg_error(fctid,"Input framelist NULL or has no input data\n");
        return(-1);
    }

    /* Initialise some things */

    vircam_lincor_init();

    /* Get parameters */

    p = cpl_parameterlist_find(parlist,"vircam.vircam_lincor.extenum");
    vircam_lincor_config.extenum = cpl_parameter_get_int(p);

    /* Sort out raw from calib frames */

    if (vircam_dfs_set_groups(framelist) != VIR_OK) {
        cpl_msg_error(fctid,"Cannot identify RAW and CALIB frames");
        vircam_lincor_tidy();
        return(-1);
    }

    /* Get the frames */

    if ((ps.labels = cpl_frameset_labelise(framelist,vircam_compare_tags,
                                           &nlab)) == NULL) {
        cpl_msg_error(fctid,"Cannot labelise the input frames");
        vircam_lincor_tidy();
        return(-1);
    }
    if ((ps.chan = vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
					      VIRCAM_CAL_CHANTAB)) == NULL) {
        cpl_msg_info(fctid,"No channel table found -- cannot continue");
	vircam_lincor_tidy();
	return(-1);
    }
    if ((ps.img = vircam_frameset_subgroup_1(framelist,ps.labels,nlab,
                                             VIRCAM_TEST_SCIENCE_RAW)) == NULL) {
        cpl_msg_info(fctid,"No raw image found -- cannot continue");
	vircam_lincor_tidy();
	return(-1);
    }

    /* Get the number of DITs */

    pp = cpl_propertylist_load(cpl_frame_get_filename(ps.img),0);
    if (vircam_pfits_get_ndit(pp,&ndit) != VIR_OK) {
        cpl_msg_error(fctid,"No value for NDIT available");
        freepropertylist(pp);
        vircam_lincor_tidy();
        return(-1);
    }
    cpl_propertylist_delete(pp);

    /* Now, how many image extensions do we want to do? If the extension
       number is zero, then we loop for all possible extensions. If it
       isn't then we just do the extension specified */

    vircam_exten_range(vircam_lincor_config.extenum,(const cpl_frame *)ps.img,
		       &jst,&jfn);
    if (jst == -1 || jfn == -1) {
	cpl_msg_error(fctid,"Unable to continue");
	vircam_lincor_tidy();
	return(-1);
    }

    /* Now loop for all the extension... */

    status = VIR_OK;
    for (j = jst; j <= jfn; j++) {
        isfirst = (j == jst);

        /* Load up the images */

        ps.imgf = vircam_fits_load(ps.img,CPL_TYPE_FLOAT,j);
        ps.chanf = vircam_tfits_load(ps.chan,j);
        if (ps.img == NULL || ps.chanf == NULL) {
	    vircam_lincor_tidy();
	    return(-1);
	}

	/* Now do the correction */

	cpl_msg_info(fctid,"Doing the linearisation for extension %d",j);
        (void)vircam_lincor(ps.imgf,ps.chanf,1,ndit,&status);
	if (status != VIR_OK) {
	    vircam_lincor_tidy();
	    return(-1);
	}

        /* Now save the result */

	cpl_msg_info(fctid,"Saving results for extension %d",j);
	if (vircam_lincor_save(framelist,parlist) != 0) {
	    vircam_lincor_tidy();
	    return(-1);
	}

	/* Tidy a few things before the next image */

	freefits(ps.imgf);
	freetfits(ps.chanf);
    }
    vircam_lincor_tidy();
    return(0);
}


/*---------------------------------------------------------------------------*/
/**
  @brief    The recipe data products are saved here
  @param    framelist    the input frame list
  @param    parlist      the input recipe parameter list
  @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int vircam_lincor_save(cpl_frameset *framelist,
			      cpl_parameterlist *parlist) {
    const char *recipeid = "vircam_lincor";
    const char *fctid = "vircam_lincor_save";
    const char *outfile = "lincor.fits";
    cpl_propertylist *plist;

    /* If we need to make a PHU then do that now based on the first frame
       in the input frame list */

    if (isfirst) {

        /* Create a new product frame object and define some tags */

        product_frame = cpl_frame_new();
        cpl_frame_set_filename(product_frame,outfile);
        cpl_frame_set_tag(product_frame,VIRCAM_PRO_SIMPLE_TEST);
        cpl_frame_set_type(product_frame,CPL_FRAME_TYPE_IMAGE);
        cpl_frame_set_group(product_frame,CPL_FRAME_GROUP_PRODUCT);
        cpl_frame_set_level(product_frame,CPL_FRAME_LEVEL_FINAL);

	/* Set up phu header */

        plist = vircam_fits_get_phu(ps.imgf);
        vircam_dfs_set_product_primary_header(plist,product_frame,framelist,
                                              parlist,(char *)recipeid,
                                              "?Dictionary?");

        /* 'Save' the PHU image */

        if (cpl_image_save(NULL,outfile,CPL_BPP_8_UNSIGNED,plist,
                           CPL_IO_DEFAULT) != CPL_ERROR_NONE) {
            cpl_msg_error(fctid,"Cannot save product PHU");
            cpl_frame_delete(product_frame);
            return(-1);
        }
        cpl_frameset_insert(framelist,product_frame);
    }

    /* Get the extension property list */

    plist = vircam_fits_get_ehu(ps.imgf);

    /* Fiddle with the header now */

    vircam_dfs_set_product_exten_header(plist,product_frame,framelist,parlist,
					(char *)recipeid,"?Dictionary?");

    /* Save the image */

    if (cpl_image_save(vircam_fits_get_image(ps.imgf),outfile,CPL_BPP_IEEE_FLOAT,
		       plist,CPL_IO_EXTEND) != CPL_ERROR_NONE) {
        cpl_msg_error(fctid,"Cannot save product image extension");
        return(-1);
    }

    return(0);
}


/*---------------------------------------------------------------------------*/
/**
  @brief    Initialise the pointers in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_lincor_init(void) {
    ps.labels = NULL;
    ps.chan = NULL;
    ps.chanf = NULL;
    ps.img = NULL;
    ps.imgf = NULL;
}


/*---------------------------------------------------------------------------*/
/**
  @brief    Free any allocated workspace in the memory structure
 */
/*---------------------------------------------------------------------------*/

static void vircam_lincor_tidy(void) {
    freespace(ps.labels);
    freefits(ps.imgf);
    freetfits(ps.chanf);
    freeframe(ps.chan);
    freeframe(ps.img);
}

/**@}*/

/*

$Log: vircam_lincor.c,v $
Revision 1.11  2007/11/26 09:56:04  jim
Fixed call to vircam_lincor to include ndit

Revision 1.10  2007/10/15 12:53:55  jim
Modified for compatibility with cpl_4.0

Revision 1.9  2007/07/09 13:22:09  jim
Modified to use new version of vircam_exten_range

Revision 1.8  2007/04/13 12:27:39  jim
Added some extra docs

Revision 1.7  2007/04/04 10:36:29  jim
Modified to use new dfs tags

Revision 1.6  2007/03/01 12:42:59  jim
Modified slightly after code checking

Revision 1.5  2006/06/15 09:58:59  jim
Minor changes to docs

Revision 1.4  2006/05/04 11:53:43  jim
Fixed _save routine so that it's more consistent with the standard CPL
way of doing things

Revision 1.3  2006/05/02 11:29:15  jim
Fixed problem where propertylist was being deleted incorrectly

Revision 1.2  2006/04/27 14:22:05  jim
Fixed docs

Revision 1.1  2006/04/24 10:42:45  jim
New routine


*/





