/* $Id: vircam_fits.c,v 1.19 2007/11/14 10:47:08 jim Exp $
 *
 * This file is part of the VIRCAM Pipeline
 * Copyright (C) 2005 Cambridge Astronomy Survey Unit
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: jim $
 * $Date: 2007/11/14 10:47:08 $
 * $Revision: 1.19 $
 * $Name:  $
 */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <math.h>
#include <string.h>

#include <cpl.h>
#include "vircam_utils.h"
#include "vircam_fits.h"

/**
    \defgroup vircam_fits vircam_fits
    \ingroup supportroutines

    \brief
    These are methods for manipulating the vircam_fits object

    \author
    Jim Lewis, CASU
*/

/**@{*/

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_fits_load
    \par Purpose:
        Load an input image into a vircam_fits object
    \par Description:
        An image from an input cpl_frame is loaded into a vircam_fits object.
	Only the image for a given extension number is loaded. The rest
	of the object properties are initialised
    \par Language:
        C
    \param frame
        The input cpl_frame object
    \param type
        The data type for the loaded image
    \param nexten
        The image extension that you want to load.
    \return
        The output vircam_fits object.
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern vir_fits *vircam_fits_load(cpl_frame *frame, cpl_type type, 
				  int nexten) {
    vir_fits *p;
    cpl_image *im;
    int nf;
    const char *fctid = "vircam_fits_load";

    /* Check for nonsense input */

    if (frame == NULL)
	return(NULL);

    /* See if you can load the image */

    im = cpl_image_load(cpl_frame_get_filename(frame),type,0,nexten);
    if (im == NULL) {
	cpl_msg_error(fctid,"Unable to load %s[%d] -- %s\n",
		      cpl_frame_get_filename(frame),nexten,
		      cpl_error_get_message());
	cpl_error_reset();
	return(NULL);
    }

    /* Get the vir_fits structure */

    p = cpl_malloc(sizeof(vir_fits));

    /* Load stuff in */

    p->image = im;
    p->nexten = nexten;
    p->phu = NULL;
    p->ehu = NULL;
    p->fname = cpl_strdup(cpl_frame_get_filename(frame));
    p->status = VIR_OK;

    /* Get the extension header and the extension name */

    (void)vircam_fits_get_ehu(p);
    if (p->ehu == NULL)
	return(NULL);
    if (cpl_propertylist_has(p->ehu,"EXTNAME")) {
        p->extname = cpl_strdup(cpl_propertylist_get_string(p->ehu,"EXTNAME"));
    } else {
	nf = 11 + (int)log10((double)nexten);
	p->extname = cpl_malloc(nf);
	(void)snprintf(p->extname,nf,"DET1.CHIP%d",nexten);
    }
    nf = strlen(p->extname) + strlen(p->fname) + 3;
    p->fullname = cpl_malloc(nf);
    (void)snprintf(p->fullname,nf,"%s[%s]",p->fname,p->extname);
   
    /* Get out of here */

    return(p);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_fits_duplicate
    \par Purpose:
        Copy a vircam_fits structure into another one.
    \par Description:
        An input vircam_fits structure is duplcated and returned
    \par Language:
        C
    \param in
        The input vircam_fits object
    \return
        The output vircam_fits object.
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern vir_fits *vircam_fits_duplicate(vir_fits *in) {
    vir_fits *p;

    /* Check for nonsense input */

    if (in == NULL)
	return(NULL);

    /* Get the vir_fits structure */

    p = cpl_malloc(sizeof(vir_fits));

    /* Now copy everything over */

    p->image = cpl_image_duplicate(in->image);
    if (in->phu != NULL) 
        p->phu = cpl_propertylist_duplicate(in->phu);
    else
	p->phu = NULL;
    if (in->ehu != NULL) 
        p->ehu = cpl_propertylist_duplicate(in->ehu);
    else
	p->ehu = NULL;
    p->fname = cpl_strdup(in->fname);
    p->extname = cpl_strdup(in->extname);
    p->fullname = cpl_strdup(in->fullname);
    p->nexten = in->nexten;
    p->status = in->status;
   
    /* Get out of here */

    return(p);
}


/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_fits_load_list
    \par Purpose:
        Load an input images into a vircam_fits object array
    \par Description:
        Images from an input cpl_frameset are loaded into a list of 
	vircam_fits objects. Only the images for a given extension number are
	loaded. The rest of the object properties are initialised
    \par Language:
        C
    \param f
        The input cpl_frameset object
    \param type
        The data type for the loaded images
    \param exten
        The image extension that you want to load.
    \return
        The output vircam_fits object list.
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern vir_fits **vircam_fits_load_list(cpl_frameset *f, cpl_type type, 
					int exten) {
    int i;
    vir_fits **p;

    /* Check for nonsense input */

    if (f == NULL)
	return(NULL);

    /* Get some workspace */

    p = cpl_malloc(cpl_frameset_get_size(f)*sizeof(vir_fits *));
    
    /* Now load each of the frames... */

    for (i = 0; i < cpl_frameset_get_size(f); i++) {
	p[i] = vircam_fits_load(cpl_frameset_get_frame(f,i),type,exten);
	if (p[i] == NULL) {
	    vircam_fits_delete_list(p,i-1);
	    return(NULL);
	}
    }

    /* Now return the array */

    return(p);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_fits_delete
    \par Purpose:
        Free all the workspace associated with a vircam_fits object
    \par Description:
        Free all the workspace associated with a vircam_fits object
    \par Language:
        C
    \param p
        The input vir_fits object
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_fits_delete(vir_fits *p) {

    /* Check for nonsense input */

    if (p == NULL)
	return;

    /* Free up workspace if it's been used */

    freeimage(p->image);
    freepropertylist(p->phu);
    freepropertylist(p->ehu);
    freespace(p->fname);
    freespace(p->extname);
    freespace(p->fullname);
    cpl_free(p);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_fits_delete_list
    \par Purpose:
        Free all the workspace associated with a list of vircam_fits objects
    \par Description:
        Free all the workspace associated with a list of vircam_fits objects
    \par Language:
        C
    \param p
        The input list of vir_fits objects
    \param n
        The number of vir_fits objects in the above array
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern void vircam_fits_delete_list(vir_fits **p, int n) {
    int i;

    /* Check for nonsense input */

    if (p == NULL)
	return;

    /* Free up workspace if it's been used */

    for (i = 0; i < n; i++)
	vircam_fits_delete(p[i]);
    freespace(p);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_fits_get_image
    \par Purpose:
        Get the CPL image from the vir_fits object
    \par Description:
        Return the CPL image from the input vir_fits object. This image is
	suitable for use in all cpl_image routines.
    \par Language:
        C
    \param p
        The input vir_fits object
    \return 
        The cpl_image object. NULL if there was an error.
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern cpl_image *vircam_fits_get_image(vir_fits *p) {
    
    /* Check for nonsense input */

    if (p == NULL)
	return(NULL);

    /* Return it */

    return(p->image);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_fits_get_nexten
    \par Purpose:
        Get the FITS extension number for the current image in a vir_fits 
	object
    \par Description:
        Get the FITS extension number for the current image in a vir_fits 
	object
    \par Language:
        C
    \param p
        The input vir_fits object
    \return 
        The extension number (-1 in case of error)
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_fits_get_nexten(vir_fits *p) {
    
    /* Check for nonsense input */

    if (p == NULL)
	return(-1);

    /* Return it */

    return(p->nexten);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_fits_get_phu
    \par Purpose:
        Get the propertylist for the primary header for a given vir_fits image.
    \par Description:
        Get the propertylist for the primary header for a given vir_fits image.
	This should only need to be read once and then can be used to add
	things to the primary header.
    \par Language:
        C
    \param p
        The input vir_fits object
    \return 
        The propertylist representing the primary header of the input image
	(NULL if there is an error).
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern cpl_propertylist *vircam_fits_get_phu(vir_fits *p) {

    /* Check for nonsense input */

    if (p == NULL)
	return(NULL);

    /* If the propertylist hasn't already been loaded, then do it now */

    if (p->phu == NULL) 
        p->phu = cpl_propertylist_load(p->fname,0);
    
    /* Return it */

    return(p->phu);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_fits_get_ehu
    \par Purpose:
        Get the propertylist for the extension header for a given vir_fits 
	image.
    \par Description:
        Get the propertylist for the extension header for a given vir_fits 
	image. This is the extension that is relevant of the image.
	This should only need to be read once and then can be used to add
	things to the primary header.
    \par Language:
        C
    \param p
        The input vir_fits object
    \return 
        The propertylist representing the extension header of the input image
	(NULL if there is an error).
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern cpl_propertylist *vircam_fits_get_ehu(vir_fits *p) {

    /* Check for nonsense input */

    if (p == NULL)
	return(NULL);

    /* If the propertylist hasn't already been loaded, then do it now */

    if (p->ehu == NULL) 
        p->ehu = cpl_propertylist_load(p->fname,p->nexten);
    
    /* Return it */

    return(p->ehu);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_fits_get_extname
    \par Purpose:
        Get the extension name for a give vir_fits object
    \par Description:
        Get the extension name for a given vir_fits object.
    \par Language:
        C
    \param p
        The input vir_fits object
    \return 
        The pointer to the extension name in the vir_fits structure -- not a 
	copy! (NULL if there is an error).
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern char *vircam_fits_get_extname(vir_fits *p) {

    /* Check for nonsense input */

    if (p == NULL)
	return(NULL);

    /* Return it */

    return(p->extname);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_fits_get_filename
    \par Purpose:
        Get the filename from which the current vir_fits object originated
    \par Description:
        Get the filename from which the current vir_fits object originated. If
	this is null, then the image didn't originate in an FITS file.
    \par Language:
        C
    \param p
        The input vir_fits object
    \return 
        The name of the file from which this image originated
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern char *vircam_fits_get_filename(vir_fits *p) {

    /* Check for nonsense input */

    if (p == NULL)
	return(NULL);

    /* Return it */

    return(p->fname);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_fits_get_fullname
    \par Purpose:
        Get the fullname of the FITS extension from which the current 
	vir_fits object originated
    \par Description:
        Get the fullname of the FITS extension  from which the current 
	vir_fits object originated. If this is null, then the image didn't 
	originate in an FITS file.
    \par Language:
        C
    \param p
        The input vir_fits object
    \return 
        The fullname name of the file from which this image originated
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern char *vircam_fits_get_fullname(vir_fits *p) {

    /* Check for nonsense input */

    if (p == NULL)
	return(NULL);

    /* Return it */

    return(p->fullname);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_fits_get_status
    \par Purpose:
        Get the error status of the current object.
    \par Description:
        Get the error status of the current object.
    \par Language:
        C
    \param p
        The input vir_fits object
    \return 
        The error status
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_fits_get_status(vir_fits *p) {
    
    /* Check for nonsense input */

    if (p == NULL)
	return(VIR_FATAL);

    /* Return it */
   
    return(p->status);
}

  
/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_fits_set_error
    \par Purpose:
        Set the error status and message for an object
    \par Description:
        The input status is checked to see if there has been a problem
	with the current object. If there has been, then the status is
	stored away and any error message from the cpl_error system is
	copied down.
    \par Language:
        C
    \param p
        The input vir_fits object
    \param status
        The input error status
    \return 
        A flag to say whether the input status was fatal.
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern int vircam_fits_set_error(vir_fits *p, int status) {

    /* Check for nonsense input */

    if (p == NULL)
	return(0);

    /* Get out of here if the status is OK */

    if (status == VIR_OK)
	return(0);

    /* Set the error status if there was an error */

    p->status = status;

    /* If there was a cpl error then spew that out now */

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
	cpl_msg_error("","%s",cpl_error_get_message());
        cpl_error_reset();
    }
    
    /* Get out of here */

    if (status == VIR_FATAL)
        return(1);
    else
	return(0);
}

/*---------------------------------------------------------------------------*/
/**
    \par Name:
        vircam_fits_wrap
    \par Purpose:
        Wrap an image in a vir_fits wrapper
    \par Description:
        The input image is inserted into a vir_fits wrapper. A model
	vir_fits object may be provided to give the new object
	headers. If the phu and ehu parameters are not null then they will
	be used as the propertylists for the new object. If not, then
	an attempt will be made to copy the propertylists from the model.
    \par Language:
        C
    \param im
        The input cpl_image
    \param model
        The input vir_fits model object
    \param phu
        The input propertylist for the extension header for the new object.
    \param ehu
        The input propertylist for the extension header for the new object.
    \return 
        The new vir_fits structure.
    \author
        Jim Lewis, CASU
 */
/*---------------------------------------------------------------------------*/

extern vir_fits *vircam_fits_wrap(cpl_image *im, vir_fits *model, 
				  cpl_propertylist *phu,
				  cpl_propertylist *ehu) {
    vir_fits *p;

    /* Check for nonsense input */

    if (im == NULL)
	return(NULL);

    /* Get the vir_fits structure */

    p = cpl_malloc(sizeof(vir_fits));

    /* Load stuff in */

    p->image = im;
    p->nexten = -1;
    if (phu != NULL) 
	p->phu = cpl_propertylist_duplicate(phu);
    else if (model != NULL) 
        p->phu = cpl_propertylist_duplicate(vircam_fits_get_phu(model));
    else
	p->phu = NULL;
    if (ehu != NULL)
        p->ehu = cpl_propertylist_duplicate(ehu);
    else if (model != NULL) 
	p->ehu = cpl_propertylist_duplicate(vircam_fits_get_ehu(model));
    else 
	p->ehu = NULL;
    p->fname = NULL;
    p->status = VIR_OK;
    p->extname = NULL;
    p->fullname = NULL;
   
    /* Get out of here */

    return(p);
}
    
/**@}*/

/*

$Log: vircam_fits.c,v $
Revision 1.19  2007/11/14 10:47:08  jim
Modified vircam_fits_duplicate so that in case the pointers to the
input headers are NULL, then this isn't an error

Revision 1.18  2007/10/25 17:34:00  jim
Modified to remove lint warnings

Revision 1.17  2007/10/19 09:25:09  jim
Fixed problems with missing includes

Revision 1.16  2007/10/15 12:50:28  jim
Modified for compatibility with cpl_4.0

Revision 1.15  2007/07/18 15:33:06  jim
Modified error message in vircam_fits_load to include extension number

Revision 1.14  2007/03/01 12:42:41  jim
Modified slightly after code checking

Revision 1.13  2007/02/20 21:13:13  jim
Better error reporting in case of load failure

Revision 1.12  2006/10/31 10:26:51  jim
Added vircam_fits_get_extname

Revision 1.11  2006/08/07 14:20:14  jim
Added vircam_fits_duplicate

Revision 1.10  2006/07/07 09:34:00  jim
wrap routine now duplicates input property lists rather than using the
input pointer

Revision 1.9  2006/07/04 09:19:05  jim
replaced all sprintf statements with snprintf

Revision 1.8  2006/05/26 15:02:21  jim
Fixed bad error message call

Revision 1.7  2006/05/24 13:34:19  jim
Added vircam_fits_wrap

Revision 1.6  2006/04/30 22:12:40  jim
Fixed memory bug

Revision 1.5  2006/04/24 13:46:36  jim
A bit more error trapping in case fits structures can't be loaded

Revision 1.4  2006/04/20 11:21:21  jim
Added _fullname method

Revision 1.3  2006/03/22 13:32:25  jim
Cosmetic changes to keep lint happy

Revision 1.2  2006/03/03 14:29:45  jim
Modified definition of vir_fits and channel table

Revision 1.1  2006/03/01 10:31:03  jim
new files


*/
