/*

$Id: imcore_radii.c,v 1.1 2005/09/13 13:25:30 jim Exp $

*/

#include <stdio.h>
#include <math.h>

#include "imcore.h"
#include "imcore_radii.h"
#include "floatmath.h"
#include "util.h"
#include "ap.h"

static float fraction (float x, float y, float r_out);
static void dchole (double a[IMNUM+1][IMNUM+1], double b[IMNUM+1], int n);

extern float imcore_exprad(float thresh, float peak, float areal0, 
			   float rcores[], int naper) {
    float pk,r_t,rad;

    /* Work out the radius... */

    pk = MAX(1.5*thresh,peak);
    r_t = sqrtf(areal0/M_PI);
    rad = 5.0*r_t/logf(pk/thresh);
    rad = MAX(r_t,MIN(5.0*r_t,MIN(rad,rcores[naper-1])));
    return(rad);
}
    
extern float imcore_kronrad(float areal0, float rcores[], float cflux[], 
			    int naper) {
    int i;
    float r_t,rad,wt;

    /* Work out the radius... */

    r_t = sqrtf(areal0/M_PI);
    rad = 0.5*rcores[0]*cflux[0];
    for (i = 1; i < naper; i++) {
	wt = MAX(0.0,cflux[i]-cflux[i-1]);
	rad += 0.5*(rcores[i] + rcores[i-1])*wt;
    }
    rad /= cflux[MIN(naper-1,7)];
    rad = MAX(r_t,MIN(5.0*r_t,MIN(2.0*rad,rcores[naper-1])));    
    return(rad);
}

extern float imcore_petrad (float areal0, float rcores[], float cflux[], 
			    int naper) {
    int j;
    float eta,r_t,etaold,r1,r2,r3,r4,r5,r_petr;

    /* Work out petrosian radius */

    r_t = sqrtf(areal0/M_PI);
    eta = 1.0;
    etaold = eta;
    j = 1;
    while (eta > 0.2 && j < naper) {
	etaold = eta;
	r1 = rcores[j]*rcores[j]/(rcores[j-1]*rcores[j-1]) - 1.0;
	r2 = cflux[j]/cflux[j-1] - 1.0;
        eta = r2/r1;
	j++;
    }
    if (j == naper) {
	r_petr = rcores[naper-1];
    } else {
	r1 = rcores[j]*rcores[j];
	r2 = rcores[j-1]*rcores[j-1];
	r3 = rcores[j-2]*rcores[j-2];
        r4 = (etaold - 0.2)/(etaold - eta);
	r5 = (0.2 - eta)/(etaold - eta);
	r_petr = r4*sqrt(0.5*(r1 + r2)) + r5*sqrt(0.5*(r2 + r3));
    }
    r_petr = MAX(r_t,MIN(5.0*r_t,MIN(2.0*r_petr,rcores[naper-1])));
    return(r_petr);
}

extern void imcore_flux(ap_t *ap, float parm[IMNUM][NPAR], int nbit, 
			float apertures[], float cflux[]) {
    double aa[IMNUM+1][IMNUM+1],bb[IMNUM+1];
    float *map,parrad[IMNUM],cn[IMNUM],xi,yi,d,a2j,a2i,di,dj,argi,argj;
    float xmin,xmax,ymin,ymax,t,xj,yj,tj,xk,yk,tk;
    unsigned char *mflag;
    long nx,ny;
    int i,ix1,ix2,iy1,iy2,ii,kk,j,k;

    /* Set up some local variables */
 
    map = ap->indata;
    mflag = ap->mflag;
    nx = ap->lsiz;
    ny = ap->csiz;

    /* Set up some convenience variables */

    for (i = 0; i < nbit; i++) {
	parrad[i] = apertures[i] + 0.5;
	cn[i] = 1.0/(M_PI*apertures[i]*apertures[i]);
    }
 
    /* Set up covariance matrix - analytic special case for cores */

    for(i = 0; i < nbit; i++) {
	aa[i][i] = cn[i];              /* overlaps totally area=pi*r**2 */
	if (nbit > 1) {
	    xi = parm[i][1];
	    yi = parm[i][2];
	    for(j = i+1; j < nbit; j++) {
		d = sqrtf((xi-parm[j][1])*(xi-parm[j][1])
		    + (yi-parm[j][2])*(yi-parm[j][2]));
		if(d >= apertures[i]+apertures[j]) {
		    aa[j][i] = 0.0;
		    aa[i][j] = aa[j][i];
		} else {
		    a2i = apertures[i]*apertures[i];
		    a2j = apertures[j]*apertures[j];
		    di = 0.5*(d + (a2i - a2j)/d);
		    dj = 0.5*(d - (a2i - a2j)/d);
		    argi = di/apertures[i];
		    argj = dj/apertures[j];
		    if (di < 0.0) {
			argi = 1.0;
			argj = 0.0;
		    }
		    if (dj < 0.0) {
			argi = 0.0;
			argj = 1.0;
		    }
		    aa[j][i] = cn[i]*cn[j]*
			(a2i*(acosf(argi) - argi*(sqrtf(1.0 - argi*argi))) +
			 a2j*(acosf(argj) - argj*(sqrtf(1.0 - argj*argj))));
		    aa[i][j] = aa[j][i];
		}
	    }
	}
    }

    /* Clear accumulators */

    for (i = 0; i < nbit; i++)
        bb[i] = 0.0;

    /* generate image-blend outer boundaries */

    xmin = 1.0e6;
    xmax = -1.0e6;
    ymin = 1.0e6;
    ymax = -1.0e6;
    for(i = 0; i < nbit; i++) {
	xi = parm[i][1];
	yi = parm[i][2];
	xmin = MIN(xmin, xi - parrad[i]);
	xmax = MAX(xmax, xi + parrad[i]);
	ymin = MIN(ymin, yi - parrad[i]);
	ymax = MAX(ymax, yi + parrad[i]);
    }
    ix1 = MAX(0,(int)xmin-1);
    ix2 = MIN(nx-1,(int)xmax);
    iy1 = MAX(0,(int)ymin-1);
    iy2 = MIN(ny-1,(int)ymax);


    /* Now go through pixel region */

    for(ii = iy1; ii <= iy2; ii++) {
	kk = ii*nx;
	for(i = ix1; i <= ix2; i++) {
	    if (mflag[kk+i] == MF_CLEANPIX || mflag[kk+i] == MF_OBJPIX) {
		t = map[kk+i];   
		for(j = 0; j < nbit; j++) {
		    xj = i - parm[j][1] + 1.0;
		    yj = ii - parm[j][2] + 1.0;
		    bb[j] += fraction(xj,yj,apertures[j])*t;
		}
	    } else {
		for (j = 0; j < nbit; j++) {
		    xj = i - parm[j][1] + 1.0;
		    yj = ii - parm[j][2] + 1.0;
		    tj = fraction(xj,yj,apertures[j]);
		    aa[j][j] -= tj*tj*cn[j]*cn[j];
		    for (k = j + 1; k < nbit; k++) {
			xk = i - parm[k][1] + 1.0;
			yk = ii - parm[k][2] + 1.0;
			tk = fraction(xk,yk,apertures[k]);
			aa[j][k] -= tk*tj*cn[j]*cn[k];
			aa[k][j] = aa[j][k];
		    }
		}
	    } 
	}
    }

    /* Supress matrix instabilities */

    if (nbit > 1) {
	for (k = 0; k < nbit; k++) {
	    for (j = k + 1; j < nbit; j++) {
		aa[j][k] = 0.99*MIN(aa[k][k],aa[j][j]);
		aa[k][j] = aa[j][k];
	    }
	}
    }

    /* Trivial solution for single object */

    if (nbit == 1) {
	cflux[0] = bb[0];
	return;
    }

    /* solve for profile intensities */

    dchole(aa,bb,nbit);
    for(i = 0; i < nbit; i++) 
	cflux[i] = cn[i]*bb[i];
}


/* returns fraction of pixel bounded by 0 -  r_out
 * x,y coordinates relative to centre
 * Uses linear approximation ok if pixel located >>1 away from centre */

static float fraction (float x, float y, float r_out) {
    float r,t,x_a,x_b,frac,tanao2,cosa,tanp2a,sqrt2o2;

    r = sqrtf(x*x + y*y);
    sqrt2o2 = 0.5*M_SQRT2;

    /* is it worth bothering? */

    if(r > r_out+sqrt2o2) 
	return(0.0);

    /* is it trivially all in? */

    if(r < r_out-sqrt2o2) 
	return(1.0);

    /* bugger - have to do some work then ... ok first ...
     * use 8-fold symmetry to convert to 0-45 degree range */

    x = fabsf(x);
    y = fabsf(y);
    if(y > x) {
	t = x;
	x = y;
	y = t;
    }

    /* If the angles are too close to cardinal points, then fudge something */

    if (x > 0.0 && y > 0.0) {
        tanao2 = 0.5*y/x;
        tanp2a = x/y;
        cosa = x/sqrt(x*x + y*y);
    } else {
        tanao2 = 0.00005;
        tanp2a = 10000.0;
        cosa = 1.0;
    }

    /* only outer radius - compute linear intersections top and bot of pixel */

    x_a = x - tanao2 + (r_out - r)/cosa;
    if(x_a < x+0.5) {

	/* intersects */

	x_b = x + tanao2 + (r_out - r)/cosa;

	/* three cases to consider */

	if(x_a < x-0.5)
	    frac = 0.5*MAX(0.0,x_b-(x-0.5))*MAX(0.0,x_b-(x-0.5))*tanp2a;
	else {
	    if(x_b > x+0.5)
		frac = 1.0 - 0.5*(x+0.5-x_a)*(x+0.5-x_a)*tanp2a;
	    else
		frac = 0.5-(x-x_a)+0.5*(x_b-x_a);
	}
    } else  /* missed entirely */
	frac = 1.0;

    return(frac);
}

/* CHOLEsky decomposition of +ve definite symmetric matrix to solve Ax = b */

static void dchole (double a[IMNUM+1][IMNUM+1], double b[IMNUM+1], int n) {
  double sum, l[IMNUM+1][IMNUM+1], y[IMNUM+1];
  double aveigv, offset;
  int i, j, k;

restart:
    l[0][0] = sqrt(a[0][0]);

    for(k = 1; k < n; k++) {
        for(j = 0; j <= k-1; j++) {
	    sum = a[j][k];
	    if(j != 0) 
		for(i = 0; i <= j-1; i++) 
		    sum -= l[i][k]*l[i][j];
	    l[j][k] = sum/l[j][j];
        }
	sum = a[k][k];
	for(i = 0; i <= k-1; i++) 
	    sum -= l[i][k]*l[i][k];
	if(sum <= 0.0) {
/* 	    fprintf(stderr, "dchole: warning: matrix ill-conditioned\n"); */
	    aveigv = a[0][0];
	    for(i = 1; i < n; i++) 
		aveigv += a[i][i];
	    /* max eigenvalue < trace */
	    offset = 0.1*aveigv/((double) n);
	    for(i = 0; i < n; i++) 
		a[i][i] += offset;
/* 	    fprintf(stderr, "dchole: Offset added to diagonal = %f\n", offset); */
	    goto restart;
	}
	l[k][k] = sqrt(sum);
    }

    /* solve Ly = b */

    y[0] = b[0]/l[0][0];
    for(i = 1; i < n; i++) {
	sum = b[i];
	for(k = 0; k <= i-1; k++) 
	    sum -= l[k][i]*y[k];
	y[i] = sum/l[i][i];
    }

    /* solve L(T)x = y */

    b[n-1] = y[n-1]/l[n-1][n-1];
    for(i = n-2; i >= 0; i--) {
	sum = y[i];
	for(k = i+1; k < n; k++) 
	    sum -= l[i][k]*b[k];
	b[i] = sum/l[i][i];
    }
}

/*

$Log: imcore_radii.c,v $
Revision 1.1  2005/09/13 13:25:30  jim
Initial entry after modifications to make cpl compliant


*/
