#!/usr/bin/zsh 

function clean_up {
    print -u2 "... cleaning up"
    rm /tmp/$$'_'*
}

trap "clean_up" EXIT

[[ -z $NEMO ]] &&  source ~/AMAS/Nemo/nemo_start.sh # We will be using some Nemo stuff...

# Values on command line

eps=0 # Softening for potentential determination
frac="0.01 0.03 0.1 0.2 0.3 0.4 0.5 0.6 0.7 0.8 0.9 0.95 0.99 1"
opening_angle=1
upper_mass_cut=1e30
imin=0
imax=1e10
isnap=1

while [[ $1 = -* ]]; do
    case $1 in
        -(e|eps|soft)   ) eps=$2; shift 2;;
        -(f|frac)   ) frac=$2; shift 2;;
	-(a|tol|theta|angle) ) opening_angle=$2; shift 2 ;;
	-(m|m_max) ) upper_mass_cut=$2; shift 2 ;;
	-imin ) imin=$2; shift 2 ;;
	-imax ) imax=$2; shift 2 ;;
	-(p|pot|keep_pot) ) pot_file=$2; shift 2 ;;
	-(t|sort_etot|etot) ) sort_etot=1; shift 1 ;;
	-nemo ) input_is_nemo=1; isnap=$2; shift 2 ;; # Input is a nemo bin file of which the snapshot $nemo_snapshot has to analysed

        *       )
              print -u2 "!!! Important message from ${0##*/}: What exactly do you mean with $1, you magnificient weirdo? !!!"
              exit 1 ;;
    esac
done

# 1) Convert to Nemo snapshot format
# 2) Use Nemo's hackcode (Tree code) to compute potential for particle sorting
# 3) Use Ellan to compute triaxiality at varius mass fraction (using potential to rank particles)

gunzip --to-stdout --force | grep -v '^#' | sort -n -k1 |\
gawk -v m_max=$upper_mass_cut -v imin=$imin -v imax=$imax '($2<m_max && $1>imin && $1<imax) {print $1,$2,$3,$4,$5,$6,$7,$8}' > /tmp/$$_data.asc

if [[ -z $input_is_nemo ]]; then
    tabtos in=/tmp/$$_data.asc out=- block1=skip,mass,pos,vel
else
    snaptrim in=- out=- times="#$isnap" XXXX need to be tested
fi |\
hackcode1 in=- out=- eps=$eps options=phi tstop=0  tol=$opening_angle |\
snapprint in=- format="%12.5e" options="m,x,y,z,vx,vy,vz,phi" |\
if [[ -z $pot_file ]]; then
    cat
else
    tee /tmp/$$_pot.asc
fi |\
run_my_ellan -pot -frac $frac ${sort_etot:+"-sort_etot"}
#echo $0 was here

# Keep potential info in file if asked
if [[ -n $pot_file ]]; then
    print -u2 "> Keeping potential information in $pot_file"
    cat /tmp/$$_data.asc | gawk '{print $1}' > /tmp/$$_names.asc
    {
	print "# Potential computed with Nemo's hackcode1 using eps=$eps tol=$opening_angle"
	print "# 1:name 2:m 3:x 4:y 5:z 6:phi"
	cat /tmp/$$_pot.asc | gawk '{print $1,$2,$3,$4,$NF}' | paste /tmp/$$_names.asc -
    } > ${pot_file%.gz}
    [[ ${pot_file##*.} == gz ]] && gzip -v ${pot_file%.gz}
fi

