#!/usr/bin/env zsh 

LC_ALL=C
export LC_ALL

exec_to_creat_xdr_file=creer_amas_rauch99 #_IntelLinux
exec_to_convert_to_asc=Amas_XDR2NCadavre #_IntelLinux

command_line="$0 $@"

# Default values 
#++++++++++++++++

# Cluster
gamma=1.75 # Central density exponent
mu=3 # Mass of central MBH relative to total mass in stars
Nstars=10000 # Number of stellar particles

iseed=261657 # random seed

# Values on command line
#++++++++++++++++++++++++

while [[ $1 = -* ]]; do
    case $1 in
	-gamma  ) gamma=$2; shift 2 ;;
	-mu  ) mu=$2; shift 2 ;;
	-Nstars ) Nstars=$2; shift 2 ;;
	-iseed  ) iseed=$2; shift 2 ;;
	-shift  ) shift_centre=1; shift 1 ;;
        -a_min  ) a_min=$2; shift 2 ;;
        -SpecialStar )
                  m_special=$2;
                  a_special=$3;
                  e_special=$4;
                  shift 4 ;;
        *       )
              print -u2 "!!! What exactly do you mean with $1, you magnificient weirdo? !!!"
              exit 1 ;;
    esac
done

# Average particle mass in Msun (ZMBAR)

m_star=1
Mstars=$(( 1.0*Nstars*m_star )) 
M_MBH=$(( 1.0*mu*Mstars ))
ZMBAR=$(( (Mstars+M_MBH)/(Nstars+1) ))

# Perform some sanity tests for paranoids

if (( 3*Mstars > M_MBH )); then
    print -u2 "!!! Total stellar mass should be at most M_MBH/3 !!!"
    exit 1
fi

if (( gamma <= 0.5 )); then
    print -u2 "!!! gamma should be larger than 0.5 !!!"
    exit 1
fi

if (( gamma >= 3 )); then
    print -u2 "!!! gamma should be lower than 3 !!!"
    exit 1
fi

# Create XDR model according to Rauch
#+++++++++++++++++++++++++++++++++++++

tmp1=/tmp/$$_tmp1
tmp2=/tmp/$$_tmp2
tmp3=/tmp/$$_tmp3

eval $exec_to_creat_xdr_file -N $Nstars -n $gamma -M_TN $mu -Name $tmp1 ${a_min:+"-a_min $a_min"} ${m_special:+"-SpecialStar $m_special $a_special $e_special"} > $tmp2 2>&1 
cat $tmp2 > _log_
Rnb_over_amax=$(( 1.0/$(cat $tmp2 | grep 'a_max in N-body units' | cut -d":" -f2 | sed 's/ //g') )) xxxx Modified 29/4/08 TEST IT
print -u2 " Rnb_over_amax = $Rnb_over_amax"
rm $tmp2

# Convert into ascii file appropriate for NBODY4 
#++++++++++++++++++++++++++++++++++++++++++++++++

eval $exec_to_convert_to_asc -NonVirialized -M_BH $mu -iseed $iseed -Mstar_avrg $m_star -StellType 5 ${shift_centre:+"-shift_center 1e-4"} $tmp1.xdr  > $tmp2 2> $tmp3
cat $tmp3 >> _log_
fact_length=$(cat $tmp3 | grep 'Enforce N-body units (factor .* on length)' | sed 's/.*(factor  *//; s/ .*//')
print -u2 " fact_length = $fact_length"
Rnb_over_amax=$(( Rnb_over_amax / fact_length ))

rm $tmp3
#rm $tmp1* $tmp3

# Write final ascii data to stdout
#++++++++++++++++++++++++++++++++++

# A header to give value of parameters and explain data in columns

print "# Initial conditions for Rauch-type stellar cluster"
print "# Generated with command line \"$command_line\" on $(hostname)"
print "# date and time : $(date)"
print "#"

print "# gamma = $gamma # Central density exponent"
print "# mu = $mu # Mass of central Msun / mass in stars"
print "# Nstars = $Nstars # Number of stellar particles"
print "# Nbody_length = $(( Rnb_over_amax )) # N-body length in units of the max semi-major axis"

[[ -n $a_min ]] && {
    print "# a_min = $a_min # minimum semi-major axis in units of the max"
}
[[ -n $m_special ]] && {
    print "# m_special = $m_special # mass of special particle (in units of total stellar mass)"
    print "# a_special = $a_special # semi-major axis of special particle (in units of max semi-major axis)"
    print "# e_special = $e_special # eccentricity of special particle"
}

print "# iseed = $iseed # random seed"

print "# RBAR = UNSPECIFIED # N-body length unit in pc"
print "# ZMBAR = $ZMBAR # Average particle mass in Msun (ARBITRARY)"
print "#"

cat $tmp2 | grep '^#' | tail -n1

cat $tmp2 | grep -v '^#' 


rm $tmp1* $tmp2 $tmp3
exit

