#!/usr/bin/env zsh 

LC_ALL=C
export LC_ALL

exec_to_creat_xdr_file=creer_amas_rauch99 #_IntelLinux
exec_to_convert_to_asc=Amas_XDR2NCadavre #_IntelLinux

command_line="$0 $@"

# Default values
#++++++++++++++++

# Cluster
gamma=1.75 # Central density exponent
M_MBH=1.0e6 # Mass of central MBH in Msun
Nstars=10000 # Number of stellar particles
a_norm=1.0 #  Semi-major axis "containing" the same amount of stellar and MBH mass, in pc 
m_star=10.0 # Mass of each stellar particle in Msun

# Particle on pre-EMRI trajectory
Porb_EMRI=1e5 # Orbital frequency in s
ecc_EMRI=0.7  # eccentricity

iseed=261657 # random seed

# Values on command line
#++++++++++++++++++++++++

while [[ $1 = -* ]]; do
    case $1 in
	-gamma  ) gamma=$2; shift 2 ;;
	-M_MBH  ) M_MBH=$2; shift 2 ;;
	-Nstars  ) Nstars=$2; shift 2 ;;
	-a_norm ) a_norm=$2; shift 2 ;;
	-m_star ) m_star=$2; shift 2 ;;
	-iseed  ) iseed=$2; shift 2 ;;
	-shift  ) shift_centre=1; shift 1 ;;
        *       )
              print -u2 "!!! What exactly do you mean with $1, you magnificient weirdo? !!!"
              exit 1 ;;
    esac
done

# Average particle mass in Msun (ZMBAR)

Mstars=$(( Nstars*m_star )) 
ZMBAR=$(( (Mstars+M_MBH)/(Nstars+1) ))

# Perform some sanity tests for paranoids

if (( 3*Mstars > M_MBH )); then
    print -u2 "!!! Total stellar mass should be at most M_MBH/3 !!!"
    exit 1
fi

if (( gamma <= 0.5 )); then
    print -u2 "!!! gamma should be larger than 0.5 !!!"
    exit 1
fi

if (( gamma >= 3 )); then
    print -u2 "!!! gamma should be lower than 3 !!!"
    exit 1
fi

# Create XDR model according to Rauch
#+++++++++++++++++++++++++++++++++++++

tmp1=/tmp/$$_tmp1
tmp2=/tmp/$$_tmp2
tmp3=/tmp/$$_tmp3
mu=$(( M_MBH/Mstars ))

R0=$(( a_norm / mu**(1/(3-gamma)) )) xxxx Modified 29/4/08 TEST IT
eval $exec_to_creat_xdr_file -N $Nstars -n $gamma -M_TN $mu -Name $tmp1 > $tmp2 2>&1 
cat $tmp2 > _log_
Rnb_over_R0=$(( 1/$(cat $tmp2 | grep 'a_max in N-body units' | cut -d":" -f2 | sed 's/ //g') )) xxxx Modified 29/4/08 TEST IT
Rnb_in_pc=$(( $Rnb_over_R0*$R0 ))
print -u2 " Rnb_over_R0 = $Rnb_over_R0; Rnb_in_pc = $Rnb_in_pc"
rm $tmp2

# Convert into ascii file appropriate for NBODY4 (assuming stars are BHs)
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

eval $exec_to_convert_to_asc -StellRad -NonVirialized -M_BH $mu -iseed $iseed -Mstar_avrg $m_star -StellType 5 ${shift_centre:+"-shift_center 1e-4"} $tmp1.xdr  > $tmp2 2> $tmp3
cat $tmp3 >> _log_
fact_length=$(cat $tmp3 | grep 'Enforce N-body units (factor .* on length)' | sed 's/.*(factor  *//; s/ .*//')
print -u2 " fact_length = $fact_length"
Rnb_in_pc=$(( $Rnb_in_pc/$fact_length ))
print -u2 " Rnb_in_pc = $Rnb_in_pc"

rm $tmp3
#rm $tmp1* $tmp3

# Write final ascii data to stdout
#++++++++++++++++++++++++++++++++++

# A header to give value of parameters and explain data in columns

print "# Initial conditions for Rauch-type stellar cluster"
print "# Generated with command line \"$command_line\" on $(hostname)"
print "# date and time : $(date)"
print "#"

print "# gamma = $gamma # Central density exponent"
print "# M_MBH = $M_MBH # Mass of central MBH in Msun"
print "# Nstars = $Nstars # Number of stellar particles"
print "# a_norm = $a_norm # Semi-major axis encompassing a stellar mass equal to the MBH's mass; in pc"
print "# R0 = $R0 # Cut-off radius in pc (max semimajor axis in Rauch Model)"
print "# Mstars = $Mstars # Mass in stellar particles in Msun"
print "# iseed = $iseed # random seed"

print "# RBAR = $Rnb_in_pc # N-body length unit in pc"
print "# ZMBAR = $ZMBAR # Average particle mass in Msun"
print "#"

cat $tmp2 | grep '^#' | tail -n1

OverMtot=1
cat $tmp2 | grep -v '^#' ##| gawk '{$1='$OverMtot'*$1;$2=$2+1e-3; print}'


#rm $tmp1* $tmp2 $tmp3
exit

