#!/bin/zsh

LC_ALL=C
export LC_ALL

exec_to_creat_xdr_file=creer_amas_rauch99
exec_to_convert_to_asc=Amas_XDR2NCadavre

command_line="$0 $@"

# Default values
#++++++++++++++++

# Cluster
gamma=1.5 # Central density exponent
M_MBH=3.5e6 # Mass of central MBH in Msun
Nstell=10000 # Number of stellar particles
R0=0.2 # Cut-off radius in pc (max semimajor axis in Rauch Model)
Mstell=1e5 # Mass in stellar particles in Msun

# Intermediate-mass couilles
m1=3e3 # Mass of inner IMBH in Msun
m2=3e3 # Mass of outer IMBH in Msun
a1=1e-3 # Semimajor axis
a2=1e-1 #
phi=0  # Orbital phase of outer IMBH (relative to inner couille)
inclination=0 # Inclination of orbital plane of outer IMBH wrt orbital plane of inner IMBH

iseed=34826458 # random seed

# Values on command line
#++++++++++++++++++++++++

while [[ $1 = -* ]]; do
   case $1 in
       -gamma   ) gamma=$2; shift 2 ;;
       -M_MBH   ) M_MBH=$2; shift 2 ;;
       -Nstell  ) Nstell=$2; shift 2 ;;
       -R0      ) R0=$2; shift 2 ;;
       -Mstell  ) Mstell=$2; shift 2 ;;
       -m1      ) m1=$2; shift 2 ;;
       -m2      ) m2=$2; shift 2 ;;
       -a1      ) a1=$2; shift 2 ;;
       -a2      ) a2=$2; shift 2 ;;
       -phi     ) phi=$2; shift 2 ;;
       -phi_deg ) phi=$(( 0.0174533*$2 )); shift 2 ;;
       -(i|inclination) ) inclination=$2; shift 2 ;;
       -i_deg   ) inclination=$(( 0.0174533*$2 )); shift 2 ;;
       -iseed  ) iseed=$2; shift 2 ;;
       *       )
             print -u2 "!!! What exactly do you mean with $1, you magnificient weirdo? !!!"
             exit 1 ;;
   esac
done

# Average particle mass in Msun (ZMBAR)

Npart=$(( Nstell+1 ))
(( m1>0 )) && Npart=$(( Npart+1 ))
(( m2>0 )) && Npart=$(( Npart+1 ))
ZMBAR=$(( ($Mstell+$M_MBH+$m1+$m2)/(1.0*$Npart) ))

# Perform some sanity tests for paranoids

if (( a1 > a2 )); then
   print -u2 "!!! a1 should be smaller than a2 !!!"
   exit 1
fi

if (( 2*a2 > R0 )); then
   print -u2 "!!! a2 should be at most 0.5*R0 !!!"
   exit 1
fi

if (( m1 > 0.2*M_MBH )); then
   print -u2 "!!! m1 should be at most 0.2*M_MBH !!!"
   exit 1
fi

if (( m2 > 0.2*M_MBH )); then
   print -u2 "!!! m2 should be at most 0.2*M_MBH !!!"
   exit 1
fi

if (( 3*Mstell > M_MBH )); then
   print -u2 "!!! Mstell should be at most M_MBH/3 !!!"
   exit 1
fi

if (( gamma <= 0.5 )); then
   print -u2 "!!! gamma should be larger than 0.5 !!!"
   exit 1
fi

if (( gamma >= 3 )); then
   print -u2 "!!! gamma should be lower than 3 !!!"
   exit 1
fi

# Create XDR model according to Rauch
#+++++++++++++++++++++++++++++++++++++

tmp1=/tmp/$$_tmp1
tmp2=/tmp/$$_tmp2
tmp3=/tmp/$$_tmp3
mu=$(( $M_MBH/$Mstell ))

eval $exec_to_creat_xdr_file -N $Nstell -n $gamma -M_TN $mu -Name $tmp1 > $tmp2 2>&1
Rnb_over_R0=$(( 1.0/$(cat $tmp2 | grep 'a_max in N-body units' | cut -d":" -f2 | sed 's/ //g') ))
Rnb_in_pc=$(( $Rnb_over_R0*$R0 ))
print -u2 " Rnb_over_R0 = $Rnb_over_R0"
rm $tmp2

# Convert into ascii file appropriate for NBODY4 (assuming stars are MS)
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

Mstar_avrg=$(( $Mstell/$Nstell ))
eval $exec_to_convert_to_asc -StellRad -NonVirialized -mu $mu -iseed $iseed -Mstar_avrg $Mstar_avrg $tmp1.xdr > $tmp2 2> $tmp3
fact_length=$(cat $tmp3 | grep 'Enforce N-body units (factor .* on length)' | sed 's/.*(factor  *//; s/ .*//')
print -u2 " fact_length = $fact_length"
Rnb_in_pc=$(( $Rnb_in_pc/$fact_length ))
print -u2 " Rnb_in_pc = $Rnb_in_pc"

rm $tmp1* $tmp3

# Insert the two IMBH (with some Vaseline(TM))
#++++++++++++++++++++++++++++++++++++++++++++++

# Compute circular velocities for IMBHs (ASSUME PURE POWER-LAW STELLAR DISTRIBUTION; ~OK if a1,a2 < R0)
# use N-body units

a1_NB=$(( $a1/$Rnb_in_pc ))
a2_NB=$(( $a2/$Rnb_in_pc ))
Mr1=$(( ($mu + ($a1/$R0)**(3-$gamma))/(1+$mu) ))
Mr2=$(( ($mu + ($a2/$R0)**(3-$gamma))/(1+$mu) ))
Vc1=$(( ( $Mr1/$a1_NB )**0.5 ))
Vc2=$(( ( $Mr2/$a2_NB )**0.5 ))

print -u2 "> Masses interior to IMBHs [N-body]: $Mr1, $Mr2"
print -u2 "> Circular velocities of IMBHs [N-body]: $Vc1, $Vc2"

# Put IMBHs on circular orbits (outer one is on an inclined orbit)

x1=$a1_NB
y1=0
z1=0
vx1=0
vy1=$Vc1
vz1=0

x2_=$( gawk 'BEGIN{print '$a2_NB'*cos('$phi'); exit}' )
y2_=$( gawk 'BEGIN{print '$a2_NB'*sin('$phi'); exit}' )
z2_=0

vx2_=$( gawk 'BEGIN{print -'$Vc2'*sin('$phi'); exit}' )
vy2_=$( gawk 'BEGIN{print  '$Vc2'*cos('$phi'); exit}' )
vz2_=0

x2=$( gawk 'BEGIN{print cos('$inclination')*'$x2_'; exit}' )
y2=$y2_
z2=$( gawk 'BEGIN{print sin('$inclination')*'$x2_'; exit}' )

vx2=$( gawk 'BEGIN{print cos('$inclination')*'$vx2_'; exit}' )
vy2=$vy2_
vz2=$( gawk 'BEGIN{print sin('$inclination')*'$vx2_'; exit}' )

# "Stellar" radius of IMBHs (in Rsun)
Rstel1=$(( 4.243e-6*$m1 ))
Rstel2=$(( 4.243e-6*$m2 ))

print -u2 "> Position of outer IMBH [N-body] : $x2, $y2, $z2"
print -u2 "> Velocity of outer IMBH [N-body] : $vx2, $vy2, $vz2"

# Insert IMBHs into ascii file
{
   # Header and central MBH
   cat $tmp2 | gawk '{print} /^[^#]/ {exit}'
   # Write data for IMBHs
   (( m1 > 0 )) && print $(( $m1/($M_MBH+$Mstell) )) $x1 $y1 $z1 $vx1 $vy1 $vz1 $Rstel1
   (( m2 > 0 )) && print $(( $m2/($M_MBH+$Mstell) )) $x2 $y2 $z2 $vx2 $vy2 $vz2 $Rstel2
   # All the bloody stars
   cat $tmp2 | gawk 'qq==1 {print} /^[^#]/ {qq=1}'
} > $tmp1

# Write final ascii data to stdout
#++++++++++++++++++++++++++++++++++

# A header to give value of parameters and explain data in columns

print "# Initial conditions for 1 or 2 IMBHs in galactic center problem (NBODY4 format)"
print "# Rauch-type stellar cluster"
print "# Generated with command line \"$command_line\" on $(hostname)"
print "# date and time : $(date)"
print "#"

print "# gamma = $gamma # Central density exponent"
print "# M_MBH = $M_MBH # Mass of central MBH in Msun"
print "# Nstell = $Nstell # Number of stellar particles"
print "# R0 = $R0 # Cut-off radius in pc (max semimajor axis in Rauch Model)"
print "# Mstell = $Mstell # Mass in stellar particles in Msun"
print "# m1 = $m1 # Mass of inner IMBH in Msun"
print "# m2 = $m2 # Mass of outer IMBH in Msun"
print "# a1 = $a1 # Semimajor axis in pc "
print "# a2 = $a2 # "
print "# phi = $phi # Orbital phase of outer IMBH (relative to inner couille) in rad"
print "# inclination = $inclination # Inclination of orbital plane of outer IMBH wrt orbital plane of inner IMBH (rad)"
print "# iseed = $iseed # random seed"

print "# RBAR = $Rnb_in_pc # N-body length unit in pc"
print "# ZMBAR = $ZMBAR # Average particle mass in Msun"
print "#"

cat $tmp1 | grep '^#' | tail -n1

# Correct N-body units to include mass of IMBHs
# may also need to shift center to avoid having MBH at 0,0,0 (not done yet)

OverMtot=$(( 1.0/(1.0 + ($m1+$m2)/($M_MBH+$Mstell)) ))
cat $tmp1 | grep -v '^#' | gawk '{$1='$OverMtot'*$1;$2=$2+1e-3; print}'


rm $tmp1* $tmp2 $tmp3
exit

