#! /bin/zsh -f
# Model for nuclear cluste with "seed" MBH

LC_ALL=C
export LC_ALL

function clean_up {
    #print -u2 "... cleaning up"
    #rm /tmp/$$'_'*
}

trap "clean_up" EXIT

command_line="$0 $@"

suff=""
DirForModels=~/AMAS/Modeles
IMF_exec=appliquerfm3
MBH_exec=Amas_IntroduceMBH

# Parameters of model

W0=6
Npart=1000000
Nstar=1000000
Rnbody_pc=2
iseed=5637
Mbh_Msun=500.0

# IMF
IMF_masses='0.08,0.5,120'
IMF_alphas='1.3,2.3'

# traitement des options sur la ligne de commande
while [[ $1 = -* ]]; do
    case $1 in
        -(w|W|W0|Wo) ) W0=$2 ; shift 2 ;;
        -Nstar ) Nstar=$2 ; shift 2 ;;
        -Npart ) Npart=$2 ; shift 2 ;;
        -(R|Rnbody) ) Rnbody_pc=$2 ; shift 2 ;;
        -(M|Mbh) ) Mbh_Msun=$2 ; shift 2 ;;
        -masses ) IMF_masses=$2 ; shift 2 ;;
        -alphas ) IMF_alphas=$2 ; shift 2 ;;

        -iseed ) iseed=$2 ; shift 2 ;;

        -NoRenorm ) no_renorm=1
	     shift 1;;
        *  )
              print -u2 "!!! option $1 illegale !!!"
              exit 1 ;;
    esac
done

# functions
#-----------

function format_number {
    nb_digits=3
    [[ -n $2 ]] && nb_digits=$2
    printf '%.'$nb_digits'g\n' $1 |\
sed 's/\./p/; s/e\([+-]\)0\([1-9]\)/e\1\2/; s/e+/e/; s/e3/k/; s/e6/M/; s/e9/G/; s/^\([^p]\)*e4/\10k/; s/^\([^p]\)*e5/\100k/;'
}



cd $DirForModels || {
    print -u2 "!!! $0: Cannot cd to $DirForModels !!!"
    exit 1
}

# Form name of files

BaseName="king_Wo$W0""_$(format_number $Npart 3)"

# Create base XDR file
#++++++++++++++++++++++

if [[ -f $BaseName.xdr ]]; then
    print -u2 "> File $BaseName.xdr exists. Using it as is."
else
    creer_king -W $W0 -N $Npart -Name $BaseName || {
	print -u2 "!!! $0: Error while running creer_king. I stop there !!!"
	exit 1
    }
fi

# Apply IMF
#++++++++++++++++++++++

FichIn=$BaseName".xdr"
FichOut=$BaseName"_NC08_IMF.xdr"
FichLog=${FichOut%.xdr}.log

if [[ -f $FichOut ]]; then
    print -u2 "> File $FichOut exists. Using it as is."
else
    print -u2  $FichIn "--> "$FichOut

    $IMF_exec ${no_renorm:+"-NoRenorm"} -iseed $iseed  -m "$IMF_masses" -a "$IMF_alphas"  $FichIn $FichOut 2>&1 | tee $FichLog || {
	print -u2 "!!! $0: Error while running $IMF_exec. I stop there !!!"
	exit 1
    }
fi

[[ -f $FichLog ]] || {
    print -u2 "!!! $0: No file $FichLog !!!"
    exit 1
}
    
AvrgStellMass=$(cat $FichLog | grep 'Masse moyenne' | head -n 1 | sed 's/.*: *//')
[[ -z $AvrgStellMass ]] && {
    print -u2 '!!! $0: cannot determine average stellar mass !!!'
    exit 1
}

Mcl_Msun=$(( AvrgStellMass*Nstar ))

# Introduce seed MBH
#++++++++++++++++++++++

Mbh_CodeUnits=$(( Mbh_Msun / Mcl_Msun ))
FichIn=$FichOut
FichOut=$BaseName"_NC08_IMF_mu$(format_number $Mbh_CodeUnits 3).xdr"
FichLog=${FichOut%.xdr}.log

if [[ -f $FichOut ]]; then
    print -u2 "> File $FichOut exists. Using it as is."
else
    print -u2  $FichIn "--> "$FichOut
    $MBH_exec -mu $Mbh_CodeUnits $FichIn || {
	print -u2 "!!! $0: Error while running $MBH_exec. I stop there !!!"
	exit 1
    }
    mv _ChangedMctr_${FichIn%.xdr}_AMAS.xdr  $FichOut
    mv _ChangedMctr_${FichIn%.xdr}_TN.xdr ${FichOut%.xdr}_TN.xdr 
    mv _ChangedMctr_${FichIn%.xdr}_ETOILES.xdr ${FichOut%.xdr}_ETOILES.xdr
    mv _ChangedMctr_${FichIn%.xdr}_MAREE.xdr ${FichOut%.xdr}_MAREE.xdr

fi


# Finalise: links and *_input_EvolAmas file
#+++++++++++++++++++++++++++++++++++++++++++++

BaseNameFinal="NC08_Wo$W0""_$(format_number $Npart 3)_Nstar$(format_number $Nstar 3)_R$(format_number $Rnbody_pc 3)_Mbh$(format_number $Mbh_Msun 3)"
print -u2 "> Creating files $BaseNameFinal_\*.xdr as symbolic links"
ln -s $FichOut $BaseNameFinal"_AMAS.xdr" 
ln -s ${FichOut%.xdr}_TN.xdr $BaseNameFinal"_TN.xdr" 
ln -s ${FichOut%.xdr}_ETOILES.xdr $BaseNameFinal"_ETOILES.xdr" 
#ln -s ${FichOut%.xdr}_MAREE.xdr $BaseNameFinal"_MAREE.xdr" 

input_file=$BaseNameFinal"_input_EvolAmas"

cat <<EOF > $input_file
#
# Parameters written by command "$command_line"
#

frac_accr_Coll       # fraction of collision-liberated gas accreted by central BH
    1.0d0   # Valeur par defaut : 1.0d0

frac_accr_Dechir     # fraction of tidal-disruption-liberated gas accreted by central BH
    0.5d0   # Valeur par defaut : 1.0d0

frac_accr_EvolSt     # fraction of stellar-evolution-liberated gas accreted by central BH
    0.0d0   # Valeur par defaut : 1.0d0

Gamma_relax          # Coef in Coulomb log
    0.01d0   # Multi-mass

MasseEtoileDef       # Average stellar mass in Msun
    $AvrgStellMass  # Valeur par defaut : 1.0d0

rNbEtoiles           # Nb (initial) total d\'etoiles
    $Nstar    # Total mass: $Mcl_Msun  Msun

TailleAmas_en_pc     # "taille" de lamas en parsec (conversion unite de taille -> pc)
    $Rnbody_pc 

M_TN_ini_def         # Default initial mass of central object
    $Mbh_CodeUnits   # $Mbh_Msun Msun
EOF

