#!/bin/zsh

# Create a simple Rauch-type cusp around a MBH with an additional shell of stars. 
# To study the diffusion of stars formed in the Galactic centre.


# example of use:
# create_CuspWithShellYoungStars.sh -mstar_cusp 0.7 -o CuspWithShell_Mcusp2e6_Mrlx0p7

LC_ALL=C
export LC_ALL
zmodload zsh/mathfunc

exec_to_creat_xdr_file=creer_amas_rauch99
exec_to_apply_imf=appliquerfm3
exec_to_merge_clusters=MergeClustersXDR
exec_to_check_cluster=checkamas

typeset -i Npart_cusp Npart_shell

M_MBH=3.6e6
Rinfl_pc=1.0
stars_per_part=1.0

M_cusp=2e6
mstar_cusp=7.0 # Effective relaxational mass around 5-8 Msun due to stellar BHs within 0.2 pc(?)
exp_dens_cusp=1.6

Nstar_shell=1e3
Rin_shell_pc=0.08
Rout_shell_pc=0.2
mstar_min_shell=1.0
mstar_max_shell=120.0
exp_IMF_shell=1.0
exp_dens_shell=3.0
p_anis_shell=0.0

base_output_name=CuspWithShell

while [[ $1 = -* ]]; do
    case $1 in # [...] cut -d '=' -f 1 | while read qu; do; print "        -$qu ) $qu=\$2; shift 2;;"; done
        -M_MBH ) M_MBH=$2; shift 2;;
        -Rinfl_pc ) Rinfl_pc=$2; shift 2;;
	-stars_per_part ) stars_per_part=$2; shift 2;;
        -M_cusp ) M_cusp=$2; shift 2;;
        -mstar_cusp ) mstar_cusp=$2; shift 2;;
        -exp_dens_cusp ) exp_dens_cusp=$2; shift 2;;
        -Nstar_shell ) Nstar_shell=$2; shift 2;;
        -Rin_shell_pc ) Rin_shell_pc=$2; shift 2;;
        -Rout_shell_pc ) Rout_shell_pc=$2; shift 2;;
        -mstar_min_shell ) mstar_min_shell=$2; shift 2;;
        -mstar_max_shell ) mstar_max_shell=$2; shift 2;;
        -exp_IMF_shell ) exp_IMF_shell=$2; shift 2;;
        -exp_dens_shell ) exp_dens_shell=$2; shift 2;;
        -p_anis_shell ) p_anis_shell=$2; shift 2;;
        -(base_output_name|o) ) base_output_name=$2; shift 2;;
        -(no_check|nc) ) no_check=1; shift 1;;
	
        *       )
              print -u2 "!!! What exactly do you mean with $1, you magnificient weirdo? !!!"
              exit 1 ;;
    esac
done

Nstar_cusp=$(( M_cusp/mstar_cusp ))
Npart_cusp=$(( Nstar_cusp/stars_per_part + 0.5 ))
Nstar_cusp=$(( Npart_cusp*stars_per_part ))
M_cusp=$(( Nstar_cusp*mstar_cusp ))
Npart_shell=$(( Nstar_shell/stars_per_part + 0.5 ))
Nstar_shell=$(( Npart_shell*stars_per_part ))

Rout_cusp_pc=$(( Rinfl_pc*(mstar_cusp*Nstar_cusp/M_MBH)**(1.0/(3.0-exp_dens_cusp)) ))

base_output_name=${base_output_name%.xdr}
base_output_name=${base_output_name%_AMAS}

[[ -f $base_output_name"_AMAS.xdr" ]] && {
    print -u2 "!!! File $output_file "$base_output_name"_AMAS.xdr exists !!!"
    exit 1
}

if (( exp_IMF_shell == 1.0 )); then
    mstar_avrg_shell=$(( (mstar_max_shell-mstar_min_shell)/log(mstar_max_shell/mstar_min_shell) ))
else
    a2=$(( 2.0-exp_IMF_shell ))
    a1=$(( 1.0-exp_IMF_shell ))
    mstar_avrg_shell=$(( a1/a2*(mstar_max_shell**a2-mstar_min_shell**a2)/(mstar_max_shell**a1-mstar_min_shell**a1) ))
fi
M_shell=$(( Nstar_shell*mstar_avrg_shell ))

#print -u2 stars_per_part = $stars_per_part
#print -u2 Npart_shell = $Npart_shell
#print -u2 Rout_cusp_pc = $Rout_cusp_pc
#print -u2 mstar_avrg_shell = $mstar_avrg_shell
 
if (( Rout_cusp_pc < Rout_shell_pc )); then
    print -u2 "!!! $!0: The cusp is smaller than the shell !!!"
    exit 1
fi

tmp1=/tmp/$$_tmp1
tmp2=/tmp/$$_tmp2
tmp3=/tmp/$$_tmp3

[[ -f _log_ ]] && rm _log_

# make the shell
############################

print -u2 "> creating shell...($Npart_shell particles)"
eval $exec_to_creat_xdr_file -CorrectSelfGrav 0 -mu $(( M_MBH/(mstar_avrg_shell*Nstar_shell) )) -a_min $(( $Rin_shell_pc/$Rout_shell_pc )) -gamma $exp_dens_shell -NbSE $Npart_shell -p $p_anis_shell -Name tmp_Shell_$$.xdr > $tmp2 2>&1 
cat $tmp2 >> _log_
amax_Nbody_shell=$(cat $tmp2 | grep 'a_max in N-body units' | cut -d":" -f2 | sed 's/ //g')
#print -u2 "amax_Nbody_shell = $amax_Nbody_shell"
rm $tmp2

eval $exec_to_apply_imf -m "$mstar_min_shell,$mstar_max_shell" -a "$exp_IMF_shell" tmp_Shell_$$.xdr tmp_Shell_IMF_$$.xdr >> _log_ 2>&1 
mv tmp_Shell_$$_TN.xdr tmp_Shell_IMF_$$_TN.xdr
rm tmp_Shell_IMF_$$_ETOILES.xdr

[[ -n $no_check ]] || eval $exec_to_check_cluster tmp_Shell_IMF_$$.xdr >> _log_

# make the background cusp
#############################

print -u2 "> creating cusp...($Npart_cusp particles)"
eval $exec_to_creat_xdr_file -CorrectSelfGrav 0 -mu $(( M_MBH/(mstar_cusp*Nstar_cusp) )) -gamma $exp_dens_cusp -NbSE $Npart_cusp -Name tmp_Cusp_$$.xdr > $tmp2 2>&1 
cat $tmp2 >> _log_
amax_Nbody_cusp=$(cat $tmp2 | grep 'a_max in N-body units' | cut -d":" -f2 | sed 's/ //g')
#print -u2 "amax_Nbody_cusp = $amax_Nbody_cusp"
rm $tmp2

[[ -n $no_check ]] || eval $exec_to_check_cluster tmp_Cusp_$$.xdr >> _log_

# merge the clusters
############################

# a type of 5 (BH) is given to the objects of the cusp to make them inert relative to stellar evolution

print -u2 "> combining cusp and shell...($(( Npart_cusp+Npart_shell )) particles)"
mu=$(( M_MBH/(mstar_avrg_shell*Nstar_shell+mstar_cusp*Nstar_cusp) ))
eval $exec_to_merge_clusters -MBHDominated -mu $mu -M2overM1 $(( mstar_avrg_shell*Nstar_shell/(mstar_cusp*Nstar_cusp))) -RescaleR2 $(( amax_Nbody_cusp/amax_Nbody_shell * Rout_shell_pc/Rout_cusp_pc )) -Mstar1 $mstar_cusp -Mstar2 $mstar_avrg_shell -Tstar1 5 -Tstar2 1 tmp_Cusp_$$.xdr tmp_Shell_IMF_$$.xdr $base_output_name > $tmp2 2>&1 
cat $tmp2 >> _log_
CoefR1=$(cat $tmp2 | grep 'Length factor for cluster 1' | cut -d":" -f2 | sed 's/ //g')
CoefR2=$(cat $tmp2 | grep 'Length factor for cluster 2' | cut -d":" -f2 | sed 's/ //g')
rm $tmp2

print -u2 "> requested amax_shell/amax_cusp = $(( Rout_shell_pc/Rout_cusp_pc ))"
print -u2 "> realised amax_shell/amax_cusp = $(( CoefR2*amax_Nbody_shell/(CoefR1*amax_Nbody_cusp) ))"

[[ -n $no_check ]] || eval $exec_to_check_cluster $base_output_name"_AMAS.xdr" >> _log_

Mcl_Msun=$(( mstar_avrg_shell*Nstar_shell+mstar_cusp*Nstar_cusp ))
Nstar=$(( Nstar_shell+Nstar_cusp ))
AvrgStellMass=$(( Mcl_Msun/Nstar ))
Rnbody_pc=$(( Rout_cusp_pc/(CoefR1*amax_Nbody_cusp) ))

cat <<EOF > $base_output_name"_input_EvolAmas"
#
# Parameters written by command "$0 $@"
#
# Total number of particles : $(( Npart_cusp+Npart_shell ))

frac_accr_Coll       # fraction of collision-liberated gas accreted by central BH
    0.0d0   # Valeur par defaut : 1.0d0

frac_accr_Dechir     # fraction of tidal-disruption-liberated gas accreted by central BH
    0.5d0   # Valeur par defaut : 1.0d0

frac_accr_EvolSt     # fraction of stellar-evolution-liberated gas accreted by central BH
    0.0d0   # Valeur par defaut : 1.0d0

Gamma_relax          # Coef in Coulomb log
    0.01d0   # Multi-mass

MasseEtoileDef       # Average stellar mass in Msun
    $AvrgStellMass  # mstar_cusp = $mstar_cusp; mstar_avrg_shell = $mstar_avrg_shell
                    # Nstar_cusp = $Nstar_cusp; Nstar_shell = $Nstar_shell
                    # M_cusp = $M_cusp; M_shell = $M_shell

rNbEtoiles           # Nb (initial) total d\'etoiles
    $Nstar    # Total mass: $Mcl_Msun  Msun

TailleAmas_en_pc     # "taille" de lamas en parsec (conversion unite de taille -> pc)
    $Rnbody_pc # Rout_cusp_pc = $Rout_cusp_pc; Rin_shell_pc = $Rin_shell_pc; Rout_shell_pc = $Rout_shell_pc

M_TN_ini_def         # Default initial mass of central object
    $mu   # $M_MBH Msun
EOF

cp _log_ $base_output_name.log

rm tmp_Cusp_$$*  tmp_Shell_$$* tmp_Shell_IMF_$$*