#!/bin/zsh

LC_ALL=C
export LC_ALL

tmp1=/tmp/$$_tmp1
tmp2=/tmp/$$_tmp2
tmp3=/tmp/$$_tmp3

exec_to_creat_xdr_file=creer_king #_IntelLinux
exec_to_get_info_model=integrer_king #_IntelLinux
exec_to_convert_to_asc=Amas_XDR2NCadavre #_IntelLinux

command_line="$0 $@"

# Default values
#++++++++++++++++

typeset -E 5 Mcl1 Mcl2 Rnb_pc_1 Rnb_pc_2 m1 m2 Rperi_pc Coef_sep
typeset -F 2 Wo1 Wo2
typeset -i Ncl1 Ncl2 iseed

# Clusters
Mcl1=3.0e4 # Mass of cluster 1 in Msun
Mcl2=3.0e4 # Mass of cluster 2 in Msun
Ncl1=60000 # Number of stellar particles in cluster 1
Ncl2=60000 # Number of stellar particles in cluster 2
Wo1=6.0 # King Wo value for cluster 1
Wo2=6.0 # King Wo value for cluster 1
Rnb_pc_1=1.0 # Size (N-body lenght scale) of cluster 1, in pc
Rnb_pc_2=1.0 # Size (N-body lenght scale) of cluster 2, in pc

# Central IMBHs
m1=300 # Mass of IMBH in cluster 1, in Msun
m2=300 # Mass of IMBH in cluster 2, in Msun

# Parabolic orbit of clusters
Rperi_pc=2.0 # Pericentre distance
Coef_sep=2.0 # Initialy, the separation between the centres is set to Coef_sep*(Rtot_1+Rtot_2)

# Other
iseed=34826458 # random seed

# Values on command line
#++++++++++++++++++++++++

while [[ $1 = -* ]]; do
    case $1 in

        -(M1|Mcl1) ) Mcl1=$2; shift 2;;
        -(M2|Mcl2) ) Mcl2=$2; shift 2;;
        -(Mcl|M)   ) Mcl1=$2; Mcl2=$2; shift 2;;
        -(N1|Ncl1) ) Ncl1=$2; shift 2;;
        -(N2|Ncl2) ) Ncl2=$2; shift 2;;
        -(Ncl|N)   ) Ncl1=$2; Ncl2=$2; shift 2;;
        -(W1|Wo1|W01) ) Wo1=$2; shift 2;;
        -(W2|Wo2|W02) ) Wo1=$2; shift 2;;
        -(W|Wo|W0)    ) Wo1=$2; Wo2=$2; shift 2;;
        -(R1|Rcl1|Rnb1|Rnb_pc_1) ) Rnb_pc_1=$2; shift 2;;
        -(R2|Rcl2|Rnb2|Rnb_pc_2) ) Rnb_pc_2=$2; shift 2;;
        -(R|Rcl|Rnb|Rnb_pc)      ) Rnb_pc_1=$2; Rnb_pc_2=$2; shift 2;;

	-(m1|Mbh1) ) m1=$2; shift 2;;
	-(m2|Mbh2) ) m2=$2; shift 2;;
	-(m|Mbh)   ) m1=$2; m2=$2; shift 2;;

	-(Rperi|Peri|peri|Rperi_pc|p|P) ) Rperi_pc=$2; shift 2;;
	-(Peri_rel|d) ) Peri_rel=$2; shift 2;;
	-(Coef_sep|C) ) Coef_sep=$2; shift 2;;

	-iseed  ) iseed=$2; shift 2 ;;

        *       )
              print -u2 "!!! What exactly do you mean with $1, you magnificient weirdo? !!!"
              exit 1 ;;
    esac
done

# Perform some sanity tests for paranoids

if (( m1 > 0.03*Mcl1 )) || (( m2 > 0.03*Mcl2 )); then
    print -u2 "!!! IMBH mass should be smaller than 3% of cluster !!!"
    exit 1
fi

# Total particle number

N=$(( Ncl1+Ncl2 ))
if (( $m1 > 0 )); then; N=$(( N + 1 )); fi
if (( $m2 > 0 )); then; N=$(( N + 1 )); fi

# Average particle mass in Msun (ZMBAR)
typeset -E 5 ZMBAR
ZMBAR=$(( (Mcl1+Mcl2+m1+m2)/N )) # Mass unit in Msun

# Decide of a conversion from N-body units (used by NBODY
# for the complete configuration of 2 clusters) and parsecs
# One says that one length unit is sqrt(Rnb_pc_1*Rnb_pc_2)
typeset -E 5 RBAR
RBAR=$(( (Rnb_pc_1*Rnb_pc_2)**0.5 )) # Length unit in pc (rather arbitrary but shouldn't matter)


# Structure parameters of clusters

typeset -E 5 Rcore1 Rtot1 Sigma1

eval $exec_to_get_info_model -sans_tab -W0 $Wo1 > $tmp1 2>/dev/null
Rcore1=$(cat $tmp1 | grep '^R_core' | sed 's/^.*: *//')
Rtot1=$(cat $tmp1 | grep '^R_tot' | sed 's/^.*: *//')
Sigma1=$(cat $tmp1 | grep '^Sigma1D_ctr' | sed 's/^.*: *//')
Rcore1=$(( Rnb_pc_1*Rcore1 ))
Rtot1=$(( Rnb_pc_1*Rtot1 ))
Sigma1=$(( 0.0655814*Sigma1*(Mcl1/Rnb_pc_1)**0.5 ))
mu1=$(( m1/Mcl1 ))

print -u2 "> Parameters of cluster 1:"
print -u2 "              W0 = $Wo1"
print -u2 "              Rcore = $Rcore1 pc"
print -u2 "              Rtot  = $Rtot1 pc"
print -u2 "              Sigma1Dctr = $Sigma1 km/s"

rm $tmp1

typeset -E 5 Rcore2 Rtot2 Sigma2
eval $exec_to_get_info_model -sans_tab -W0 $Wo2 > $tmp1 2>/dev/null || {
    print -u2 "!!! failed to run $exec_to_get_info_model !!!"
    exit 1
}
Rcore2=$(cat $tmp1 | grep '^R_core' | sed 's/^.*: *//')
Rtot2=$(cat $tmp1 | grep '^R_tot' | sed 's/^.*: *//')
Sigma2=$(cat $tmp1 | grep '^Sigma1D_ctr' | sed 's/^.*: *//')
Rcore2=$(( Rnb_pc_2*Rcore2 ))
Rtot2=$(( Rnb_pc_2*Rtot2 ))
Sigma2=$(( 0.0655814*Sigma2*(Mcl2/Rnb_pc_2)**0.5 ))
mu2=$(( m2/Mcl2 ))

[[ -n $Peri_rel ]] && Rperi_pc=$(( Peri_rel*(Rtot1+Rtot2) ))

print -u2 "> Parameters of cluster 2:"
print -u2 "              W0 = $Wo2" 
print -u2 "              Rcore = $Rcore2 pc"
print -u2 "              Rtot  = $Rtot2 pc"
print -u2 "              Sigma1Dctr = $Sigma2 km/s"

rm $tmp1

# Create King models
#++++++++++++++++++++

# Create XDR file
eval $exec_to_creat_xdr_file -W $Wo1 -N $Ncl1 -Name $tmp1 -iseed $iseed >/dev/null 2>&1 || {
    print -u2 "!!! failed to run $exec_to_creat_xdr_file !!!"
    exit 1
}
iseed=$(( iseed+7 ))

# Convert into ascii file appropriate for NBODY4 (assuming stars are MS)
# introduce central IMBH
ascfile1=/tmp/clust1_$$.asc
ascfile2=/tmp/clust2_$$.asc

Mstar_avrg=$(( Mcl1/Ncl1 ))

eval $exec_to_convert_to_asc -StellRad -NonVirialized -mu $mu1 -iseed $iseed -Mstar_avrg $Mstar_avrg $tmp1.xdr > $ascfile1 2> $tmp3 || {
    print -u2 "!!! failed to run $exec_to_convert_to_asc !!!"
    exit 1
}
iseed=$(( iseed+7 ))
fact_length=$(cat $tmp3 | grep 'Enforce N-body units (factor .* on length)' | sed 's/.*(factor  *//; s/ .*//')

rm $tmp1* $tmp3

### Rnb_pc_1=$(( $Rnb_pc_1*$fact_length ))
print -u2 "> Rnb_pc_1 = $Rnb_pc_1"

if (( Wo2 == Wo1 )) && (( Ncl1 == Ncl2 )) && (( mu1 == mu2 )); then
    ln -s $ascfile1 $ascfile2
else
    eval $exec_to_creat_xdr_file -W $Wo2 -N $Ncl2 -Name $tmp1 -iseed $iseed > /dev/null 2>&1 || {
	print -u2 "!!! failed to run $exec_to_creat_xdr_file !!!"
	exit 1
    }
    iseed=$(( iseed+7 ))

    Mstar_avrg=$(( Mcl2/Ncl2 ))

    eval $exec_to_convert_to_asc -StellRad -NonVirialized -mu $mu2 -iseed $iseed -Mstar_avrg $Mstar_avrg $tmp1.xdr > $ascfile2 2> $tmp3
    iseed=$(( iseed+7 ))
    fact_length=$(cat $tmp3 | grep 'Enforce N-body units (factor .* on length)' | sed 's/.*(factor  *//; s/ .*//')

    rm $tmp1* $tmp3
fi

### Rnb_pc_2=$(( $Rnb_pc_2*$fact_length ))
print -u2 "> Rnb_pc_2 = $Rnb_pc_2"

# Compute parabolic orbit of clusters
#+++++++++++++++++++++++++++++++++++++

# Initial positions and velocities of the clusters' centres
# in code units (Mtot=1, G=1)
lambda1=$(( (Mcl1+m1)/(Mcl1+Mcl2+m1+m2) ))
lambda2=$(( 1 -lambda1 ))
d=$(( Coef_sep*(Rtot1+Rtot2)/RBAR )) # separation (code units)
Vrel=$(( (2/d)**0.5 ))
dmin=$(( Rperi_pc/RBAR ))
Vmax=$(( (2/dmin)**0.5 ))
l=$(( Vmax*dmin )) # specific angular momentum
dx=$(( 2*dmin -d ))
dy=$(( (4*dmin*(dmin-dx))**0.5 ))

# Position of clusters
Xcl1=$(( lambda2*dx ))
Ycl1=$(( lambda2*dy ))
Xcl2=$(( -lambda1*dx ))
Ycl2=$(( -lambda1*dy ))

Vrelx=$(( l*dy/d**2*(1+(1+(d/dy)**2*((dx*Vrel/l)**2-1))**0.5) ))
Vrely=$(( -((Vrel**2 - Vrelx**2)**0.5) ))

print -u2 "> relative velocity in NBODY units: $Vrelx; $Vrely"
# Velocity of clusters
VXcl1=$(( lambda2*Vrelx ))
VYcl1=$(( lambda2*Vrely ))
VXcl2=$(( -lambda1*Vrelx ))
VYcl2=$(( -lambda1*Vrely ))

# Put the clusters on orbit and combine them into one file (in stdout)
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

# To convert to nemo binary format, do something like
#  grep -v '^#' < data_nbody.asc > _tmp_ && tabtos in=_tmp_ out=data_nemo.bin block1=mass,pos,vel

# A header to give value of parameters and explain data in columns

typeset -E 5 Vmax_kms Vrel_kms

Vmax_kms=$(( 0.0655814*Vmax*((Mcl1+Mcl2+m1+m2)/RBAR)**0.5 ))
Vrel_kms=$(( 0.0655814*Vrel*((Mcl1+Mcl2+m1+m2)/RBAR)**0.5 ))

print "# Initial conditions for 2 clusters with IMBHs (NBODY4 format)"
print "# King models"
print "# Generated with command line \"$command_line\" by $(whoami) on $(hostname)"
print "# date and time : $(date)"
print "#"

print "### Clusters"
print "# Mcl1=$Mcl1 # Mass of cluster 1 in Msun"
print "# Mcl2=$Mcl2 # Mass of cluster 2 in Msun"
print "# Ncl1=$Ncl1 # Number of stellar particles in cluster 1"
print "# Ncl2=$Ncl2 # Number of stellar particles in cluster 2"
print "# Wo1=$Wo1 # King Wo value for cluster 1"
print "# Wo2=$Wo2 # King Wo value for cluster 1"
print "# Rnb_pc_1=$Rnb_pc_1 # Size (N-body length scale) of cluster 1, in pc"
print "# Rnb_pc_2=$Rnb_pc_2 # Size (N-body lenght scale) of cluster 2, in pc"
print "#"
print "# Rcore1=$Rcore1 # core radius of cluster 1, in pc"
print "# Rcore2=$Rcore2 # core radius of cluster 2, in pc"
print "# Rtot1=$Rtot1 # truncation radius of cluster 1, in pc"
print "# Rtot2=$Rtot2 # truncation radius of cluster 2, in pc"
print "# Sigma1=$Sigma1 # Central velocity dispersion of cluster 1 (1D) in km/s"
print "# Sigma2=$Sigma2 # Central velocity dispersion of cluster 2 (1D) in km/s"
print "#"

print "### Central IMBHs"
print "# m1=$m1 # Mass of IMBH in cluster 1, in Msun"
print "# m2=$m2 # Mass of IMBH in cluster 2, in Msun"
print "#"

print "### Parabolic orbit of clusters"
print "# Rperi_pc=$Rperi_pc # Pericentre distance in pc (point-mass dynamics)"
print "# Vmax=$Vmax_kms # Relative velocity at pericenter (point-mass dynamics) in km/s"
print "# Coef_sep=$Coef_sep # Initialy, the separation between the centres is set to Coef_sep*(Rtot1+Rtot2)"
print "# Vrel=$Vrel_kms # Initial relative velocity in km/s"
print "#"

print "#### Other"
print "# iseed=$iseed # random seed"
print "#"

print "# RBAR = $RBAR # N-body length unit in pc"
print "# ZMBAR = $ZMBAR # Average particle mass in Msun"
print "#"

cat $ascfile1 | grep '^#' | tail -n1 # column names

# Cluster 1
factM=$lambda1
factR=$(( Rnb_pc_1/RBAR ))
factV=$(( (factM/factR)**0.5 ))
cat $ascfile1 | gawk -v factM=$factM -v factR=$factR -v factV=$factV -v Xcl=$Xcl1 -v Ycl=$Ycl1 -v VXcl=$VXcl1 -v VYcl=$VYcl1 '
    BEGIN{OFS=" "}
    /^[^#]/ {
	m=$1;x=$2;y=$3;z=$4;vx=$5;vy=$6;vz=$7;Rstar=$8;
 	m=sprintf("%19.12e",factM*m);
	x=sprintf("%19.12e",factR*x+Xcl);    y=sprintf("%19.12e",factR*y+Ycl);     z=sprintf("%19.12e",factR*z); 
	vx=sprintf("%19.12e",factV*vx+VXcl); vy=sprintf("%19.12e",factV*vy+VYcl); vz=sprintf("%19.12e",factV*vz);
	print m,x,y,z,vx,vy,vz,Rstar}
'

# Cluster 2
factM=$lambda2
factR=$(( Rnb_pc_2/RBAR ))
factV=$(( (factM/factR)**0.5 ))
cat $ascfile2 | gawk -v factM=$factM -v factR=$factR -v factV=$factV -v Xcl=$Xcl2 -v Ycl=$Ycl2 -v VXcl=$VXcl2 -v VYcl=$VYcl2 '
    BEGIN{OFS=" "}
    /^[^#]/ {
	m=$1;x=$2;y=$3;z=$4;vx=$5;vy=$6;vz=$7;Rstar=$8;
 	m=sprintf("%19.12e",factM*m);
	x=sprintf("%19.12e",factR*x+Xcl);     y=sprintf("%19.12e",factR*y+Ycl);    z=sprintf("%19.12e",factR*z); 
	vx=sprintf("%19.12e",factV*vx+VXcl); vy=sprintf("%19.12e",factV*vy+VYcl); vz=sprintf("%19.12e",factV*vz);
	print m,x,y,z,vx,vy,vz,Rstar}
'

rm $ascfile1 $ascfile2	
