#!/bin/zsh

LC_ALL=C
export LC_ALL

tmp1=/tmp/$$_tmp1
tmp2=/tmp/$$_tmp2
tmp3=/tmp/$$_tmp3

exec_to_creat_xdr_file=creer_king #_IntelLinux
exec_to_get_info_model=integrer_king #_IntelLinux
exec_to_convert_to_asc=Amas_XDR2NCadavre #_IntelLinux

command_line="$0 $@"

# Default values
#++++++++++++++++

typeset -E 5 Mcl Rnb_pc m
typeset -F 2 Wo
typeset -i Ncl iseed

# Clusters
Mcl=3.0e4 # Mass of cluster in Msun
Ncl=60000 # Number of stellar particles
Wo=6.0 # King Wo value
Rnb_pc=1.0 # Size (N-body lenght scale) of cluster, in pc

# Central IMBHs
m=300 # Mass of central IMBH, in Msun

# Other
iseed=34826458 # random seed

# Values on command line
#++++++++++++++++++++++++

while [[ $1 = -* ]]; do
    case $1 in
        -(Mcl|M)   ) Mcl=$2; shift 2;;
        -(Ncl|N)   ) Ncl=$2; shift 2;;
        -(W|Wo|W0) ) Wo=$2; shift 2;;
        -(R|Rcl|Rnb|Rnb_pc) ) Rnb_pc=$2; shift 2;;

	-(m|Mbh)   ) m=$2; shift 2;;

	-iseed  ) iseed=$2; shift 2 ;;

        *       )
              print -u2 "!!! What exactly do you mean with $1, you magnificient weirdo? !!!"
              exit 1 ;;
    esac
done

# Perform some sanity tests for paranoids

if (( m > 0.03*Mcl )); then
    print -u2 "!!! IMBH mass should be smaller than 3% of cluster !!!"
    ## exit 1
fi

# Total particle number

N=$(( Ncl ))
if (( $m > 0 )); then; N=$(( N + 1 )); fi

# Average particle mass in Msun (ZMBAR)
typeset -E 10 ZMBAR
ZMBAR=$(( (Mcl+m)/N )) # Mass unit in Msun

# Decide of a conversion from N-body units (used by NBODY
# for the complete configuration of 2 clusters) and parsecs
# One says that one length unit is sqrt(Rnb_pc_1*Rnb_pc_2)
typeset -E 10 RBAR

# Structure parameters of cluster

typeset -E 5 Rcore Rtot Sigma

eval $exec_to_get_info_model -sans_tab -W0 $Wo > $tmp1 2>/dev/null || {
    print -u2 "!!! failed to run $exec_to_get_info_model !!!"
    exit 1
}
Rcore=$(cat $tmp1 | grep '^R_core' | sed 's/^.*: *//')
Rtot=$(cat $tmp1 | grep '^R_tot' | sed 's/^.*: *//')
Sigma=$(cat $tmp1 | grep '^Sigma1D_ctr' | sed 's/^.*: *//')
Rcore=$(( Rnb_pc*Rcore ))
Rtot=$(( Rnb_pc*Rtot ))
Sigma=$(( 0.0655814*Sigma*(Mcl/Rnb_pc)**0.5 ))
mu=$(( m/Mcl ))

print -u2 "> Parameters of cluster:"
print -u2 "              W0 = $Wo"
print -u2 "              Rcore = $Rcore pc"
print -u2 "              Rtot  = $Rtot pc"
print -u2 "              Sigma1Dctr = $Sigma km/s"

rm $tmp1

# Create King model
#+++++++++++++++++++

# Create XDR file
# echo $exec_to_creat_xdr_file -W $Wo -N $Ncl -Name $tmp1 -iseed $iseed 
eval $exec_to_creat_xdr_file -W $Wo -N $Ncl -Name $tmp1 -iseed $iseed 1>&2 || {
    print -u2 "!!! failed to run $exec_to_creat_xdr_file !!!"
    exit 1
}
iseed=$(( iseed+7 ))

# Convert into ascii file appropriate for NBODY4 (assuming stars are MS)
# introduce central IMBH
ascfile=/tmp/clust_$$.asc

Mstar_avrg=$(( Mcl/Ncl ))

eval $exec_to_convert_to_asc -StellRad -NonVirialized -mu $mu -iseed $iseed -Mstar_avrg $Mstar_avrg $tmp1.xdr > $ascfile 2> $tmp3 || {
    print -u2 "!!! failed to run $exec_to_convert_to_asc !!!"
    exit 1
}
iseed=$(( iseed+7 ))
fact_length=$(cat $tmp3 | grep 'Enforce N-body units (factor .* on length)' | sed 's/.*(factor  *//; s/ .*//')
print -u2 "> Factor on length to enforce N-body units : $fact_length"

rm $tmp1* $tmp3

RBAR=$(( $Rnb_pc/$fact_length ))
print -u2 "> RBAR = $RBAR" # Length unit in pc


# Write ascii data with extended header info to stdout
#+++++++++++++++++++++++++++++++++++++++++++++++++++++

# To convert to nemo binary format, do something like
#  grep -v '^#' < data_nbody.asc > _tmp_ && tabtos in=_tmp_ out=data_nemo.bin block1=mass,pos,vel

# A header to give value of parameters and explain data in columns

typeset -E 5 Vmax_kms Vrel_kms

Vmax_kms=$(( 0.0655814*Vmax*((Mcl+m)/RBAR)**0.5 ))
Vrel_kms=$(( 0.0655814*Vrel*((Mcl+m)/RBAR)**0.5 ))

print "# Initial conditions for a cluster with central IMBH (NBODY4 format)"
print "# King model"
print "# Generated with command line \"$command_line\" by $(whoami) on $(hostname)"
print "# date and time : $(date)"
print "#"

print "# Mcl=$Mcl # Mass of cluster in Msun"
print "# Ncl=$Ncl # Number of stellar particles in cluster"
print "# Wo=$Wo # King Wo value"
print "# Rnb_pc=$Rnb_pc # Size (N-body length scale before introduction of IMBH), in pc"
print "#"
print "# Rcore=$Rcore # core radius of cluster, in pc"
print "# Rtot=$Rtot # truncation radius of cluster, in pc"
print "# Sigma=$Sigma # Central velocity dispersion of cluster (1D) in km/s"
print "#"

print "# m=$m # Mass of IMBH, in Msun"
print "#"

print "# iseed=$iseed # random seed"
print "#"

print "# RBAR = $RBAR # N-body length unit in pc"
print "# ZMBAR = $ZMBAR # Average particle mass in Msun"
print "#"

cat $ascfile | grep '^#' | tail -n1 # column names

# Cluster data
factM=1
factR=$(( Rnb_pc/RBAR ))
factV=$(( 1/factR**0.5 ))
cat $ascfile | gawk -v factR=$factR -v factV=$factV -v shift_pos=1e-4 '
    BEGIN{OFS=" "}
    /^[^#]/ {
	m=$1;x=$2;y=$3;z=$4;vx=$5;vy=$6;vz=$7;Rstar=$8;
 	m=sprintf("%19.12e",m);
	x=sprintf("%19.12e",factR*x+shift_pos); y=sprintf("%19.12e",factR*y+shift_pos); z=sprintf("%19.12e",factR*z+shift_pos); 
	vx=sprintf("%19.12e",factV*vx);         vy=sprintf("%19.12e",factV*vy);         vz=sprintf("%19.12e",factV*vz);
	print m,x,y,z,vx,vy,vz,Rstar}
'

rm $ascfile
