#! /usr/bin/perl -w

use Getopt::Long;
Getopt::Long::Configure("default");

my $command_line=$0." ".join(" ",@ARGV);;

# Deafult IMF (piecewise power-law; by default Salpeter)
my $IMF_m_string="0.2,120";
my $IMF_a_string="2.35"; # Salpeter exponent

my $Nbin=10;         # Number of mass bins
my $delta_discr=1.0; # Discretisation exponent. If >1: more bins at low mass. If <1: more bins a high mass

my $spedi_NOL=5; # Plummer model by default
my $spedi_RSC=1; # size of 1 pc by default
my $spedi_XNTOT=1e6;

# Get options on command-line (if any)

GetOptions(
      "spedi"              => \$spedi,        # Output in spedi's kug1.dat format
      "LeeFP"              => \$LeeFP,        # For use with H.M Lee's FP code
      "linear"             => \$linear,       # Use linear mass scale (instead of logarithmic)
      "equal_number|en"    => \$equal_number, # Equal number of stars per bin
      "equal_mass|em"      => \$equal_mass,   # Equal mass per bin
      "NOL=i"              => \$spedi_NOL,    # Model type for spedi
      "RSC=f"              => \$spedi_RSC,    # Model radius for spedi (Plummer radius if Plummer)
      "XNTOTA=f"           => \$spedi_XNTOT,  # Total number of stars (spedi)
      "masses|M=s"         => \$IMF_m_string, # list of limit masses, like "-masses=0.1,0.5,2,20,100"
      "alphas|A=s"         => \$IMF_a_string, # list of IMF exponents (dN/dM \propto M^{-exponent}), like "-alphas=1.5,3,2.3,4"
      "delta=f"            => \$delta_discr,  # determines how mass bins are spaced (see below)
      "nbin|ncomp=i"       => \$Nbin          # number of bins (=components)
     ) || die;

# Use of -delta=xx
####################
# Let M_DMC_i be the individual mass of stars of DMC (discretized mass component) number i
# then:
# lg(M_DMC_i)=lg(M_min)+lg(M_max/M_min)*((i-1)/(Ncomp-1))**delta i=1..Ncomp
#
# delta allows to put more DMCs at low masses (delta>1) or at high masses (delta<1), 
# delta=1 gives the log equal spacing.

# Kroupa Initial Mass Function
###############################
# -masses=0.08,0.5,1,120 -alphas=1.3,2.2,2.7 # From 1993 paper (MNRAS) Effective IMF for Galactic field
# -masses=0.01,0.08,0.5,120 -alphas=0.3,1.3,2.3 # From MNRAS 322, 231 (2001) IMF in large clusters


my @IMF_m=split /[,|:;]/, $IMF_m_string;
my @IMF_a=split /[,|:;]/, $IMF_a_string;

$RSC_str=sprintf('%11.5e',$spedi_RSC);
$RSC_str=~s/(e|E)([-+])/${2}0/;
$XNTOT_str=sprintf('%11.5e',$spedi_XNTOT);
$XNTOT_str=~s/(e|E)([-+])/${2}0/;

die("!!! The number of IMF exponents has to match the number of mass limits minus one !!!") 
  unless ($#IMF_m==$#IMF_a+1);

my @IMF_c=0*@IMF_a;
my @IMF_fracN=0*@IMF_m; # Cumulative number fraction
my @IMF_fracM=0*@IMF_m; # Cumulative mass fraction
$IMF_fracN[0]=0.0;
$IMF_fracM[0]=0.0;
my $Mstar_avrg;
my $Mstar_rlx_avrg;
init_MF();

print_info() unless ($spedi or $LeeFP);

my $Mbin_inf=$IMF_m[0];
my $Mbin_sup;
my $Mavrg_bin;
my $Nfrac_bin;
my $Mfrac_bin;
my $lMinf=log($IMF_m[0]);
my $Minf=$IMF_m[0];
my $Dlm=log($IMF_m[$#IMF_m]/$IMF_m[0]);
my $Dm=$IMF_m[$#IMF_m]-$IMF_m[0];
print "# 1: Mbin      2: Minf      3: N_frac    4: M_frac\n" unless ($spedi or $LeeFP);

if ($spedi) {
  #print "\n";
  #print "\n";
}

if ($LeeFP) {
  printf "%3i\n", $Nbin;
}

$iIMF=0;
$frac=0;
for ($ibin=1; $ibin<=$Nbin; $ibin++) {
  if ($equal_number or $equal_mass) {
    $frac = $ibin/$Nbin;
    if ($equal_number) {
      while ($frac>$IMF_fracN[$iIMF]) {$iIMF++;}
      if ($iIMF==0) {
	$frac_prev = 0.0;
      } else {
	$frac_prev = $IMF_fracN[$iIMF-1];
      }
      $m_prev = $IMF_m[$iIMF];
      $alph1=1-$IMF_a[$iIMF];
      $Mbin_sup= ( ($IMF_m[$iIMF+1]**$alph1-$m_prev**$alph1)*($frac-$frac_prev)/($IMF_fracN[$iIMF]-$frac_prev) + $m_prev**$alph1 )**(1.0/$alph1);
    } else {
      while ($frac>$IMF_fracM[$iIMF]) {$iIMF++;}
      if ($iIMF==0) {
	$frac_prev = 0.0;
      } else {
	$frac_prev = $IMF_fracM[$iIMF-1];
      }
      $m_prev = $IMF_m[$iIMF];
      $alph2=2-$IMF_a[$iIMF];
      $Mbin_sup= ( ($IMF_m[$iIMF+1]**$alph2-$m_prev**$alph2)*($frac-$frac_prev)/($IMF_fracM[$iIMF]-$frac_prev) + $m_prev**$alph2 )**(1.0/$alph2);
    }
  } else {
    if ($linear) {
      $Mbin_sup=$Minf+$Dm*($ibin/$Nbin)**$delta_discr;
    } else {
      $Mbin_sup=exp($lMinf+$Dlm*($ibin/$Nbin)**$delta_discr);
    }
  }
  ( $Mavrg_bin, $Nfrac_bin ) = compute_bin($Mbin_inf,$Mbin_sup);
  $Mfrac_bin = $Nfrac_bin*$Mavrg_bin/$Mstar_avrg;
  if ($spedi) {
    $MIND_str=sprintf('%11.5e',$Mavrg_bin);
    $MIND_str=~s/(e|E)([-+])/${2}0/;
    $MTOT_str=sprintf('%11.5e',$Mavrg_bin*$Nfrac_bin*$spedi_XNTOT);
    $MTOT_str=~s/(e|E)([-+])/${2}0/;
    print sprintf('%2d',$ibin),"   MTOT= ",$MTOT_str," MIND= ",$MIND_str,"  MREM= 0.60000+000 TAU= 1.00000+020\n";
    print "     RTOT= 2.23000+002 RIND= 1.00000+000  TIND= 1.00000+000 RSC= ",$RSC_str,"\n";
    print "      EXP= 0.00000+000 UINI=-1.00000-005 XBIND= 1.00000+000 NOL= ",sprintf('%3d',$spedi_NOL),"\n";
    print "      LEQ=  T  T  T  T  T  T\n";
  } elsif ($LeeFP) {
    print sprintf(" %12.5e"x4,$Mavrg_bin,$Mbin_inf,$Nfrac_bin,$Mfrac_bin),"\n"; # only column 1 and 3 will be used
  } else {
    print sprintf(" %12.5e"x4,$Mavrg_bin,$Mbin_inf,$Nfrac_bin,$Mfrac_bin),"\n";
  }
  $Mbin_inf=$Mbin_sup;
}

if ($spedi) {
  print "XNTOTA= ",$XNTOT_str,"  XIMF= 0.00000+000 LEQ=  T\n";
  print "813,2,44,813,2,45,814,2,44,814,2,45,814,3,44,814,3,45,0,0\n";
  print "\n";
  print "\n";
}

if ($spedi or $LeeFP) {
  print "\n";
  print_info();
}

if ($LeeFP) {
  print "#\n";
  print "# columns are\n";
  print "#             Mbin Minf N_frac M_frac\n";
  print "# but only column 1 and 3 are used\n";
}

#========== SUBROUTINES ==========

sub print_info { # print info...
  print "# IMF discretisation\n";
  print "# Command line: ",$command_line,"\n";
  print "# IMF limit masses : ",join(",",@IMF_m),"\n";
  print "# IMF exponents    : ",join(",",@IMF_a),"\n";
  print "# Discretisation exponent : ",$delta_discr,"\n";
  print "# Number of bins          : ",$Nbin,"\n";
  print "# Average mass            : ",sprintf("%12.5e",$Mstar_avrg),"\n";
  print "# Average relaxation mass : ",sprintf("%12.5e",$Mstar_rlx_avrg),"\n";
}

sub init_MF { # compute continuity coefficients of IMF (normalized to 1)
  #print STDERR "> entering init_MF\n";
  $IMF_c[0]=1;
  for ($iIMF=1; $iIMF<$#IMF_m; $iIMF++) {
    $IMF_c[$iIMF]=$IMF_c[$iIMF-1]*$IMF_m[$iIMF]**($IMF_a[$iIMF]-$IMF_a[$iIMF-1]);
  }
  # normalisation
  my $norm=0;
  $Mstar_avrg=0;
  $Mstar_rlx_avrg=0;
  for ($iIMF=0; $iIMF<$#IMF_m; $iIMF++) {
    $a1=1-$IMF_a[$iIMF];
    $norm=$norm            +$IMF_c[$iIMF]/$a1*($IMF_m[$iIMF+1]**$a1-$IMF_m[$iIMF]**$a1);
    $a2=2-$IMF_a[$iIMF];
    $Mstar_avrg=$Mstar_avrg+$IMF_c[$iIMF]/$a2*($IMF_m[$iIMF+1]**$a2-$IMF_m[$iIMF]**$a2);
    $a3=3-$IMF_a[$iIMF];
    $Mstar_rlx_avrg=$Mstar_rlx_avrg+$IMF_c[$iIMF]/$a3*($IMF_m[$iIMF+1]**$a3-$IMF_m[$iIMF]**$a3);
  }
  $Mstar_rlx_avrg=$Mstar_rlx_avrg/$Mstar_avrg;
  $Mstar_avrg=$Mstar_avrg/$norm;
  my $fracM_prec=0.0;
  my $fracN_prec=0.0;
  for ($iIMF=0; $iIMF<=$#IMF_c; $iIMF++) {
    $IMF_c[$iIMF]=+$IMF_c[$iIMF]/$norm;
    $a1=1-$IMF_a[$iIMF];
    $a2=2-$IMF_a[$iIMF];
    $IMF_fracN[$iIMF] = $fracN_prec + $IMF_c[$iIMF]/$a1               *($IMF_m[$iIMF+1]**$a1-$IMF_m[$iIMF]**$a1);
    $IMF_fracM[$iIMF] = $fracM_prec + $IMF_c[$iIMF]/($Mstar_avrg*$a2) *($IMF_m[$iIMF+1]**$a2-$IMF_m[$iIMF]**$a2);
    $fracN_prec = $IMF_fracN[$iIMF];
    $fracM_prec = $IMF_fracM[$iIMF];
  }
  #print STDERR "< exiting init_MF\n";
}

sub compute_bin { # Compute average stellar mass and number fraction between $Mmin and $Mmax
  my ($Mmin, $Mmax) = @_;
  #print STDERR "> entering compute_bin ",$Mmin," ",$Mmax,"\n";
  return (0.5*($Mmin+$Mmax), 0.0) if ($Mmax<=$Mmin);
  return (0.5*($Mmin+$Mmax), 0.0) if ($Mmin>$IMF_m[$#IMF_m]);
  return (0.5*($Mmin+$Mmax), 0.0) if ($Mmax<$IMF_m[0]);

  my $iIMF=0;
  while ($IMF_m[$iIMF+1]<$Mmin) {
    $iIMF++;
  }

  # integrate
  my $Mslice=0.0;
  my $Nslice=0.0;
  my $Minf=0.0;
  my $Msup=0.0;
  while ($Msup<$Mmax && $iIMF<$#IMF_m) {
    $Minf=($Mmin>$IMF_m[$iIMF])   ? $Mmin : $IMF_m[$iIMF];
    $Msup=($Mmax<$IMF_m[$iIMF+1]) ? $Mmax : $IMF_m[$iIMF+1];
    $a2=2-$IMF_a[$iIMF];
    $Mslice=$Mslice+$IMF_c[$iIMF]/$a2*($Msup**$a2-$Minf**$a2);
    $a1=1-$IMF_a[$iIMF];
    $Nslice=$Nslice+$IMF_c[$iIMF]/$a1*($Msup**$a1-$Minf**$a1);
    $iIMF++;
  }
  #print STDERR $Mslice,"/",$Nslice,"\n";
  #print STDERR "> exiting compute_bin\n";
  return ($Mslice/$Nslice, $Nslice);
}
