#!/bin/zsh

LC_ALL=C
export LC_ALL

# Load math function module
zmodload zsh/mathfunc

source phys_units.sh -d

tmp1=/tmp/$$_tmp1
tmp2=/tmp/$$_tmp2
tmp3=/tmp/$$_tmp3

exec_to_creat_xdr_file=creer_king #_IntelLinux
exec_to_get_info_model=integrer_king #_IntelLinux
exec_to_convert_to_asc=Amas_XDR2NCadavre #_IntelLinux

command_line="$0 $@"

# Default values
#++++++++++++++++

typeset -E 10 Mcl Rnb_pc m eps_star eps_MBH Rmp_min Rmp_min gamma_mp eps_mp m_mp
typeset -F 2 Wo
typeset -i Ncl Nmp iseed

# Stellar cluster
Mcl=1.0e8 # Mass of cluster in Msun
Ncl=256000 # Number of stellar particles
Wo=9.0 # King Wo value
Rnb_pc=30.0 # Size (N-body lenght scale) of cluster, in pc
eps_star=0.0 # softening length of stellar particles, in pc

# Central MBH
m=3.6e6 # Mass of central IMBH, in Msun
eps_MBH=0.03 # softening length, in pc

# Massive perturbers
Nmp=10 # number
m_mp=1.0e6 # individual mass, in Msun
Rmp_min=5.0 # minimum distance in pc
Rmp_max=50.0 # maximum distance in pc
gamma_mp=2.0 # exponent of density law
eps_mp=0.1 # softening length, in pc

# Other
iseed=34826458 # random seed

# Values on command line
#++++++++++++++++++++++++

while [[ $1 = -* ]]; do
    case $1 in
        -(Mcl|Mstar|M) ) Mcl=$2; shift 2;;
        -(Ncl|Nstar|N) ) Ncl=$2; shift 2;;
        -(W|Wo|W0) ) Wo=$2; shift 2;;
        -(R|Rcl|Rnb|Rnb_pc) ) Rnb_pc=$2; shift 2;;
	-eps_star ) eps_star=$2; shift 2;;

	-(m|Mbh|M_MBH)    ) m=$2; shift 2;;
	-(eps_MBH|eps_bh) ) eps_MBH=$2; shift 2;;

	-Nmp       ) Nmp=$2; shift 2;;
	-m_mp      ) m_mp=$2; shift 2;;
	-eps_mp    ) eps_mp=$2; shift 2;;
	-Rmp_min   ) Rmp_min=$2; shift 2;;
	-Rmp_max   ) Rmp_max=$2; shift 2;;
	-gamma_max ) gamma_max=$2; shift 2;;

	-iseed  ) iseed=$2; shift 2 ;;

        *       )
              print -u2 "!!! What exactly do you mean with $1, you magnificient weirdo? !!!"
              exit 1 ;;
    esac
done

# Perform some sanity tests for paranoids

if (( m > 0.03*Mcl )); then
    print -u2 "!!! IMBH mass should be smaller than 3% of cluster !!!"
    ## exit 1
fi

# Total particle number

N=$(( Ncl + Nmp ))
if (( $m > 0 )); then; N=$(( N + 1 )); fi

# Average particle mass in Msun (ZMBAR)
typeset -E 10 ZMBAR
ZMBAR=$(( (Mcl+m+Nmp*Mmp)/N )) # Mass unit in Msun

# Decide of a conversion from N-body units

typeset -E 10 RBAR
RBAR=$Rnb_pc # Is taht strictly correct? Who cares?

# Structure parameters of cluster

typeset -E 5 Rcore Rtot Sigma

eval $exec_to_get_info_model -sans_tab -W0 $Wo > $tmp1 2>/dev/null || {
    print -u2 "!!! failed to run $exec_to_get_info_model !!!"
    exit 1
}
Rcore=$(cat $tmp1 | grep '^R_core' | sed 's/^.*: *//')
Rtot=$(cat $tmp1 | grep '^R_tot' | sed 's/^.*: *//')
Sigma=$(cat $tmp1 | grep '^Sigma1D_ctr' | sed 's/^.*: *//')
Rcore=$(( Rnb_pc*Rcore ))
Rtot=$(( Rnb_pc*Rtot ))
Sigma=$(( 0.0655814*Sigma*(Mcl/Rnb_pc)**0.5 ))
mu=$(( m/Mcl ))

print -u2 "> Parameters of cluster:"
print -u2 "              W0 = $Wo"
print -u2 "              Rcore = $Rcore pc"
print -u2 "              Rtot  = $Rtot pc"
print -u2 "              Sigma1Dctr = $Sigma km/s"

rm $tmp1

# Create King model
#+++++++++++++++++++

# Create XDR file
# echo $exec_to_creat_xdr_file -W $Wo -N $Ncl -Name $tmp1 -iseed $iseed 
eval $exec_to_creat_xdr_file -W $Wo -N $Ncl -Name $tmp1 -iseed $iseed 1>&2 || {
    print -u2 "!!! failed to run $exec_to_creat_xdr_file !!!"
    exit 1
}
iseed=$(( iseed+7 ))

# Convert into ascii file appropriate for NBODY4 (assuming stars are MS)
# introduce central IMBH
ascfile=/tmp/clust_$$.asc

Mstar_avrg=$(( Mcl/Ncl ))

eval $exec_to_convert_to_asc -KeepUnits -NonVirialized -mu $mu -iseed $iseed -Mstar_avrg $Mstar_avrg $tmp1.xdr > $ascfile 2> $tmp3 || {
    print -u2 "!!! failed to run $exec_to_convert_to_asc !!!"
    exit 1
}
iseed=$(( iseed+7 ))
#fact_length=$(cat $tmp3 | grep 'Enforce N-body units (factor .* on length)' | sed 's/.*(factor  *//; s/ .*//')
#print -u2 "> Factor on length to enforce N-body units : $fact_length"

rm $tmp1* $tmp3

#RBAR=$(( $Rnb_pc/$fact_length ))
RBAR=$Rnb_pc
print -u2 "> RBAR = $RBAR" # Length unit in pc

# Add the massive pertubers (as particles 2-1+Nmp)
# Don't sweat over dynamical equilibrium and enforcing cluster size, etc... 
# Just make sure that the total mass is 1

typeset -E 15 m_mp_nb R_mp V_mp x_mp y_mp z_mp vx_mp vy_mp vz_mp

pi=$(( 4*atan(1) ))
m_mp_nb=$(( m_mp/(m+Mcl+Nmp*m_mp) ))

Rmp_file=/tmp/rmp_$$.asc
beta=$(( 3.0-gamma_mp ))
for (( i_mp=1; i_mp<=Nmp; i_mp++ )); do
    # Radius of orbit
    R_mp=$(( ( (Rmp_max**beta-Rmp_min**beta)*rand48(seedrand48) + Rmp_min**beta )**(1.0/beta) ))
    echo $(( R_mp / Rnb_pc ))
done | sort -g | integrer_king -W0 $Wo -i 2>/dev/null | gawk '/^[^#]/ {print $1,$2}' |\
while read R_mp Mr; do
    V_mp=$(( ((Mr+$m/$Mcl)/R_mp)**0.5 )) # Circular velocity

    # Random position on sphere of radius R_mp
    phi=$(( 2.0*pi*rand48(seedrand48) ))
    cosphi=$(( cos(phi) ))
    sinphi=$(( sin(phi) ))
    sintheta=$(( 1.0-2.0*rand48(seedrand48) ))
    costheta=$(( (1.0-sintheta**2)**0.5 ))

    x_mp=$(( R_mp*costheta*cosphi ))
    y_mp=$(( R_mp*costheta*sinphi ))
    z_mp=$(( R_mp*sintheta ))


    # Random orientation of velocity, perp to radius-vector
    tau=$(( 2.0*pi*rand48(seedrand48) ))
    costau=$(( cos(tau) ))
    sintau=$(( sin(tau) ))

    vx_mp=$(( V_mp*(costau*sinphi    + sintau*sintheta*cosphi) ))
    vy_mp=$(( V_mp*(costau*(-cosphi) + sintau*sintheta*sinphi) ))
    vz_mp=$(( V_mp*(costau*0         + sintau*(-costheta)    ) ))

    print $m_mp_nb $x_mp $y_mp $z_mp $vx_mp $vy_mp $vz_mp $((eps_mp/Rnb_pc))
done  > $Rmp_file


# Write ascii data with extended header info to stdout
#+++++++++++++++++++++++++++++++++++++++++++++++++++++

# To convert to nemo binary format, do something like
#  grep -v '^#' < data_nbody.asc > _tmp_ && tabtos in=_tmp_ out=data_nemo.bin block1=mass,pos,vel

# A header to give value of parameters and explain data in columns

typeset -E 5 Vmax_kms Vrel_kms

Vmax_kms=$(( 0.0655814*Vmax*((Mcl+m)/RBAR)**0.5 ))
Vrel_kms=$(( 0.0655814*Vrel*((Mcl+m)/RBAR)**0.5 ))

print "# Initial conditions for simulation of Galactic centre with massive perturbers"
print "# King model"
print "# Generated with command line \"$command_line\" by $(whoami) on $(hostname)"
print "# date and time : $(date)"
print "#"

print "# Mcl=$Mcl # Mass of stellar cluster in Msun"
print "# Ncl=$Ncl # Number of stellar particles in cluster"
print "# Wo=$Wo # King Wo value"
print "# Rnb_pc=$Rnb_pc # Size (N-body length scale before introduction of MBH and perturbers), in pc"
print "# eps_star=$eps_star # Softening length of stellar particles, in pc"
print "#"
print "# Rcore=$Rcore # core radius of cluster, in pc"
print "# Rtot=$Rtot # truncation radius of cluster, in pc"
print "# Sigma=$Sigma # Central velocity dispersion of cluster (1D) in km/s"
print "#"

print "# m=$m # Mass of central MBH, in Msun"
print "# eps_MBH=$eps_MBH # softening length, in pc"
print "#"

print "# Nmp=$Nmp #  number of massive pertubers"
print "# m_mp=$m_mp # Individual mass, in Msun"
print "# Rmp_min=$Rmp_min # minimum distance in pc"
print "# Rmp_max=$Rmp_max # maximum distance in pc"
print "# gamma_mp=$gamma_mp # exponent of density law"
print "# eps_mp=$eps_mp # softening length, in pc"
print "#"

print "# iseed=$iseed # random seed"
print "#"

print "# RBAR = $RBAR # N-body length unit in pc"
print "# ZMBAR = $ZMBAR # Average particle mass in Msun"
print "#"
print "# 1:m 2:x 3:y 4:z 5:vx 6:vy 7:vz 8:eps_soft"

# Cluster data
factM=$(( Mcl/(m+Mcl+Nmp*m_mp) ))

# Central MBH 
cat $ascfile | gawk -v factM=$factM -v eps=$eps_MBH '/^[^#]/ {printf "%22.15e %22.15e %22.15e %22.15e %22.15e %22.15e %22.15e %12.5e\n", $1*factM,$2,$3,$4,$5,$6,$7,eps; exit} '

# massive perturbers
cat $Rmp_file  | gawk '{printf "%22.15e %22.15e %22.15e %22.15e %22.15e %22.15e %22.15e %12.5e\n", $1,$2,$3,$4,$5,$6,$7,$8} '

# stellar particles
cat $ascfile | gawk -v factM=$factM -v eps=$eps_star '/^[^#]/ {if (i++>0){printf "%22.15e %22.15e %22.15e %22.15e %22.15e %22.15e %22.15e %12.5e\n", $1*factM,$2,$3,$4,$5,$6,$7,eps}} '

rm $ascfile $Rmp_file
