#!/usr/bin/env zsh

LC_ALL=C
export LC_ALL

tmp1=/tmp/$$_tmp1
tmp2=/tmp/$$_tmp2
tmp3=/tmp/$$_tmp3

exec_to_creat_xdr_file=creer_king #_IntelLinux
exec_to_get_info_model=integrer_king #_IntelLinux
exec_to_convert_to_asc=Amas_XDR2NCadavre #_IntelLinux

command_line="$0 $@"

# Default values
#++++++++++++++++

typeset -E 5 Mcl Rnb_pc m Rperi_pc Coef_sep
typeset -F 2 Wo d0_pc Vinfty_kms
typeset -i Ncl iseed

# Cluster
Mcl=1.0e5 # Mass of cluster in Msun
Ncl=100000 # Number of stellar particles in cluster
Wo=6.0 # King Wo value for cluster
Rnb_pc=1.0 # Size (N-body lenght scale) of cluster, in pc

# Central IMBH
m=10000 # Mass of IMBH, in Msun

# Hyperbolic orbit of cluster
Rperi_pc=0.0 # Pericentre distance assuming point-masses
Coef_sep=2.0 # Initialy, the separation between the centres is set to Coef_sep*Rtot_cluster
Vinfty_kms=100 # Relative velocity at infinity in km/s

# Other
iseed=34826458 # random seed

# Values on command line
#++++++++++++++++++++++++

while [[ $1 = -* ]]; do
    case $1 in

        -(M|Mcl)             ) Mcl=$(( 1.0*$2 )); shift 2;;
        -(Ncl|N)             ) Ncl=$2; shift 2;;
        -(W|Wo|W0)           ) Wo=$(( 1.0*$2 )); shift 2;;
        -(R|Rcl|Rnb|Rnb_pc)  ) Rnb_pc=$(( 1.0*$2 )); shift 2;;

	-(m|Mbh)             ) m=$2; shift 2;;

	-(Rperi|Peri|peri|Rperi_pc|p|P|dmin) ) Rperi_pc=$(( 1.0*$2 )); shift 2;;
	-(Sep_pc|d0|d)       ) d0_pc=$((   1.0*$2 )); shift 2;;
	-(Coef_sep|C)        ) Coef_sep=$((   1.0*$2 )); shift 2;;
	-(V|Vrel|Vinfty)     ) Vinfty_kms=$(( 1.0*$2 )); shift 2;;

	-iseed  ) iseed=$2; shift 2 ;;

        *       )
              print -u2 "!!! What exactly do you mean with $1, you magnificient weirdo? !!!"
              exit 1 ;;
    esac
done

# Perform some sanity tests for paranoids

# Total particle number

N=$(( Ncl+1 ))

# Average particle mass in Msun (ZMBAR)
typeset -E 5 ZMBAR
ZMBAR=$(( (Mcl+m)/N )) # Mass unit in Msun

# Decide of a conversion from N-body units (used by NBODY
# for the complete configuration of 2 clusters) and parsecs
# One says that one length unit is sqrt(Rnb_pc_1*Rnb_pc_2)
typeset -E 5 RBAR
RBAR=$Rnb_pc # Length unit in pc


# Structure parameters of cluster

typeset -E 5 Rcore Rtot Sigma

eval $exec_to_get_info_model -sans_tab -W0 $Wo > $tmp1 2>/dev/null
Rcore=$(cat $tmp1 | grep '^R_core' | sed 's/^.*: *//')
Rtot=$(cat $tmp1 | grep '^R_tot' | sed 's/^.*: *//')
Sigma=$(cat $tmp1 | grep '^Sigma1D_ctr' | sed 's/^.*: *//')
Rcore=$(( Rnb_pc*Rcore ))
Rtot=$(( Rnb_pc*Rtot ))
Sigma=$(( 0.0655814*Sigma*(Mcl/Rnb_pc)**0.5 ))

print -u2 "> Parameters of cluster:"
print -u2 "              W0 = $Wo"
print -u2 "              Rcore = $Rcore pc"
print -u2 "              Rtot  = $Rtot pc"
print -u2 "              Sigma1Dctr = $Sigma km/s"

rm $tmp1

# Create King model
#+++++++++++++++++++

# Create XDR file
eval $exec_to_creat_xdr_file -W $Wo -N $Ncl -Name $tmp1 -iseed $iseed >/dev/null 2>&1 || {
    print -u2 "!!! failed to run $exec_to_creat_xdr_file !!!"
    exit 1
}
iseed=$(( iseed+7 ))

# Convert into ascii file appropriate for NBODY4 (assuming stars are MS)
# introduce central IMBH
ascfile1=/tmp/clust1_$$.asc
ascfile2=/tmp/clust2_$$.asc

Mstar_avrg=$(( Mcl/Ncl ))

eval $exec_to_convert_to_asc -StellRad -NonVirialized -iseed $iseed -Mstar_avrg $Mstar_avrg $tmp1.xdr > $ascfile1 2> $tmp3 || {
    print -u2 "!!! failed to run $exec_to_convert_to_asc !!!"
    exit 1
}
iseed=$(( iseed+7 ))
fact_length=$(cat $tmp3 | grep 'Enforce N-body units (factor .* on length)' | sed 's/.*(factor  *//; s/ .*//')

rm $tmp1* $tmp3

### Rnb_pc=$(( $Rnb_pc*$fact_length ))
print -u2 "> Rnb_pc = $Rnb_pc"

# Compute hyperbolic orbit (put cluster and IMBH at prescibed distance of each other)
#++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

# Initial positions and velocities of the clusters' centres
# in code units (Mtot=1, G=1)
lambda1=$(( Mcl/(Mcl+m) ))
lambda2=$(( 1 - lambda1 ))
if (( d0_pc > 0 )); then
    Coef_sep=$(( d0_pc/Rtot ))
fi
d=$(( Coef_sep*Rtot/RBAR )) # initial separation (code units)

One_kms=$(( 1.5248228563e+01*(RBAR/(Mcl+m))**0.5 ))
Vinfty=$(( Vinfty_kms*One_kms ))
a=$(( 1/Vinfty**2 ))

if (( Rperi_pc > 0.0 )); then
    dmin=$(( Rperi_pc/RBAR ))
    b=$(( dmin*(1+a/dmin)**0.5 ))
    Vmax=$(( (Vinfty**2+2/dmin)**0.5 ))
    l=$(( Vmax*dmin )) # specific angular momentum
    tan_theta2=$(( a/b ))
    dx=$(( (tan_theta2*(b-tan_theta2*d) -(d*d-b*b+2*d*b*tan_theta2)**0.5)/(1+tan_theta2**2) ))
    dy=$(( (d**2-dx**2)**0.5 ))
else
    l=0.0
    dx=$(( -d ))
    dy=0.0
fi
# print $dx $dy  $(( d**2-dx**2 -dy**2 ))>> _www

# Positions
X1=$(( lambda2*dx ))
Y1=$(( lambda2*dy ))
X2=$(( -lambda1*dx ))
Y2=$(( -lambda1*dy ))

Vrel=$(( (Vinfty**2+2/d)**0.5 ))
if (( Rperi_pc > 0.0 )); then
    Vrelx=$(( l*dy/d**2*(1+(1+(d/dy)**2*((dx*Vrel/l)**2-1))**0.5) ))
    Vrely=$(( -((Vrel**2 - Vrelx**2)**0.5) ))
else
    Vrelx=$Vrel
    Vrely=0.0
fi

print -u2 "> relative velocity in NBODY units: $Vrelx; $Vrely"
# Velocity of clusters
VX1=$(( lambda2*Vrelx ))
VY1=$(( lambda2*Vrely ))
VX2=$(( -lambda1*Vrelx ))
VY2=$(( -lambda1*Vrely ))

# Put the cluster and IMBH on orbit and combine them into one file (in stdout)
#+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

# To convert to nemo binary format, do something like
#  grep -v '^#' < data_nbody.asc > _tmp_ && tabtos in=_tmp_ out=data_nemo.bin block1=mass,pos,vel

# A header to give value of parameters and explain data in columns

typeset -E 5 Vmax_kms Vrel_kms

Vmax_kms=$(( Vmax/One_kms ))
Vrel_kms=$(( Vrel/One_kms ))

print "# Initial conditions for cluster with impacting IMBH (NBODY4 format)"
print "# King model"
print "# Generated with command line \"$command_line\" by $(whoami) on $(hostname)"
print "# date and time : $(date)"
print "#"

print "### Cluster"
print "# Mcl=$Mcl # Mass in Msun"
print "# Ncl=$Ncl # Number of stellar particles in cluster"
print "# Wo=$Wo # King Wo value"
print "# Rnb_pc=$Rnb_pc # Size (N-body length scale) of cluster, in pc"
print "#"
print "# Rcore=$Rcore # core radius, in pc"
print "# Rtot=$Rtot # truncation radius, in pc"
print "# Sigma=$Sigma # Central velocity dispersion (1D) in km/s"
print "#"

print "### IMBH"
print "# m=$m # Mass of IMBH, in Msun"
print "#"

print "### Hyperbolic orbit of cluster"
print "# Vinfty_kms=$Vinfty_kms # Relative velocity at infinity in km/s"
print "# Rperi_pc=$Rperi_pc # Pericentre distance in pc (point-mass dynamics)"
print "# Vmax=$Vmax_kms # Relative velocity at pericenter (point-mass dynamics) in km/s"
print "# Coef_sep=$Coef_sep # Initialy, the separation between the centres is set to Coef_sep*(Rtot1+Rtot2)"
print "# Vrel=$Vrel_kms # Initial relative velocity in km/s"
print "#"

print "#### Other"
print "# iseed=$iseed # random seed"
print "#"

print "# RBAR = $RBAR # N-body length unit in pc"
print "# ZMBAR = $ZMBAR # Average particle mass in Msun"
print "#"

cat $ascfile1 | grep '^#' | tail -n1 # column names

# IMBH
Mstar=$(( 2.1216190926e-06*m )) # Schwarzschild radius in solar radius
printf '%19.12e %19.12e %19.12e %19.12e %19.12e %19.12e %19.12e %19.12e\n' $lambda2 $X2 $Y2 0 $VX2 $VY2 0 $Mstar

# Cluster
factM=$lambda1
factR=$(( Rnb_pc/RBAR ))
factV=$(( (factM/factR)**0.5 ))
cat $ascfile1 | gawk -v factM=$factM -v factR=$factR -v factV=$factV -v Xcl=$X1 -v Ycl=$Y1 -v VXcl=$VX1 -v VYcl=$VY1 '
    BEGIN{OFS=" "}
    /^[^#]/ {
	m=$1;x=$2;y=$3;z=$4;vx=$5;vy=$6;vz=$7;Rstar=$8;
 	m=sprintf("%19.12e",factM*m);
	x=sprintf("%19.12e",factR*x+Xcl);    y=sprintf("%19.12e",factR*y+Ycl);     z=sprintf("%19.12e",factR*z); 
	vx=sprintf("%19.12e",factV*vx+VXcl); vy=sprintf("%19.12e",factV*vy+VYcl); vz=sprintf("%19.12e",factV*vz);
	print m,x,y,z,vx,vy,vz,Rstar}
'

rm $ascfile1
