#!/usr/bin/zsh
# Combine information from Splotch*_lagr_ref*.asc[.gz] file into sequential files indicating evolution
# of Lagrangian quantities; one file per mass fraction

function clean_up {
    print -u2 "... cleaning up"
    rm /tmp/$$'_'*
}

trap "clean_up" EXIT


[[ -f NT_Splotchs.asc ]] || [[ -f NT_Splotchs.asc.gz ]] || {
    print u2 "!!! File NT_Splotchs.asc[.gz] not found. So long, looser !!!"
    exit 1
}

# There might be different series of *_lagr_* files, for different reference particle or particle sets
# For instance, in the case of a merger, one can have series relative to each cluster's central particle
# and one series relative to the center of the merged cluster

Refs=( $(ls Splotch*_lagr_ref*.asc(|.gz) | sed 's/^.*ref//; s/\.asc.*//' | sort -u) )

print -u2 "> Following reference(s) found : $Refs"

gunzip --to-stdout --force NT_Splotchs.asc(|.gz) | grep -v '^#' > /tmp/$$_NT

for ref in $Refs; do 
    
    Files=( $(ls Splotch*_lagr_ref$ref.asc(|.gz) | sort) )
    
#   Get mass fractions
    Fracs=( $(gunzip --to-stdout --force $Files[0] | gawk '/^[^#]/ {printf "%6.5f\n", $1 }' ) )

#   Form header
    print "# Evolution of Lagrangian stuff" > /tmp/$$_hdr
    {
	echo step\\ntime
	gunzip --to-stdout --force $Files[0] | grep '^#' | tail -n 1 | sed 's/^#//; s/ *[0-9][0-9]* *: */ /g; s/^  *//; s/  *$//' | tr ' ' '\012'
    } | gawk '{print ++i":"$1}' | tr '\012' ' ' | sed 's/^/# /' >>  /tmp/$$_hdr
    echo >> /tmp/$$_hdr

    ref_file=EvolLagr_ref$ref""_reference.asc
    echo "# Evolution of reference point for Lagrangian quantities" > $ref_file
    echo "# 1:step_ref 2:time_ref 3:m_ref 4:x_ref 5:y_ref 6:z_ref 7:vx_ref 8:vy_ref 9:vz_ref" >> $ref_file
    
#   Accumulate data into files; one for each ref and mass fraction
    for file in $Files; do
	gunzip --to-stdout --force $file | grep -v '^#' > /tmp/$$_data
	step=$(( ${${file%_lagr*}##*_} ))
        time=$(cat /tmp/$$_NT | gawk -v step=$step '$1==step {print $2}')
	echo $step" "$time$(gunzip --to-stdout --force $file | grep '^# .*_ref  *=' | cut -d '=' -f 2 | tr '\012' ' ' | sed 's/  */ /g') >> $ref_file
	for frac in $Fracs; do
	    print -u2 " Extracting info for mass fraction $frac from $file (step $step)..."
	    file_evol=EvolLagr_ref$ref""_$(echo $frac | sed 's/\./p/').asc
	    [[ -f $file_evol ]] || cp  /tmp/$$_hdr $file_evol
	    print -u2 "File $file_evol"
	    cat /tmp/$$_data | gawk -v frac=$frac -v step=$step -v time=$time '$1==frac {print step,time,$0}' >> $file_evol
	done
    done
done

