#! /usr/bin/perl -w
# Some constants
use constant iType_MS => 1;
use constant iType_WD => 3;
use constant iType_NS => 4;
use constant iType_BH => 5;

use Getopt::Long;
Getopt::Long::Configure("default");

my $command_line=$0." ".join(" ",@ARGV);;

# Deafult IMF (piecewise power-law; by default Salpeter)
my $IMF_m_string="0.2,120";
my $IMF_a_string="2.35"; # Salpeter exponent

my $Nbin=10;         # Number of mass bins
my $delta_discr=1.0; # Discretisation exponent. If >1: more bins at low mass. If <1: more bins a high mass

my $spedi_NOL=5; # Plummer model by default
my $spedi_RSC=1; # size of 1 pc by default
my $spedi_XNTOT=1e6;

my $M_TO=1; # Turn-off mass
my $M_WD=0.6; # White dwarf mass
my $M_NS=1.4; # Neutron star mass
my $M_BH=10.0; # Stellar BH mass
my $Mmax_WD=8.0; # Max MS mass to form WD
my $Mmax_NS=30.0; # Max MS mass to form NS

# Get options on command-line (if any)

GetOptions(
      "spedi"              => \$spedi,        # Output in spedi's kug1.dat format
      "linear"             => \$linear,       # Use linear mass scale (instead of logarithmic)
      "NOL=i"              => \$spedi_NOL,    # Model type for spedi
      "RSC=f"              => \$spedi_RSC,    # Model radius for spedi (Plummer radius if Plummer)
      "XNTOTA=f"           => \$spedi_XNTOT,  # Total number of stars (spedi)
      "masses|M=s"         => \$IMF_m_string, # list of limit masses, like "-masses=0.1,0.5,2,20,100"
      "alphas|A=s"         => \$IMF_a_string, # list of IMF exponents (dN/dM \propto M^{-exponent}), like "-alphas=1.5,3,2.3,4"
      "delta=f"            => \$delta_discr,  # determines how mass bins are spaced (see below) on the MS
      "nbin|ncomp=i"       => \$Nbin,         # number of bins (=components) for the MS
      "M_TO=f"             => \$M_TO,
      "M_WD=f"             => \$M_WD,
      "M_NS=f"             => \$M_NS,
      "M_BH=f"             => \$M_BH,
      "Mmax_WD=f"          => \$Mmax_WD,
      "Mmax_NS=f"          => \$Mmax_NS
     ) || die;

# Use of -delta=xx
####################
# Let M_DMC_i be the individual mass of stars of DMC (discretized mass component) number i
# then:
# lg(M_DMC_i)=lg(M_min)+lg(M_max/M_min)*((i-1)/(Ncomp-1))**delta i=1..Ncomp
#
# delta allows to put more DMCs at low masses (delta>1) or at high masses (delta<1), 
# delta=1 gives the log equal spacing.

# In addition to the $Nbin components for the MS, three components are created for 
# the WDs, NSs and stellar BHs

# Kroupa Initial Mass Function
###############################
# -masses=0.08,0.5,1,120 -alphas=1.3,2.2,2.7 # From 1993 paper (MNRAS)
# -masses=0.01,0.08,0.5,120 -alphas=0.3,1.3,2.3 # from MNRAS 322, 231 (2001); for galactic field


my @IMF_m=split /[,|:;]/, $IMF_m_string;
my @IMF_a=split /[,|:;]/, $IMF_a_string;

$RSC_str=sprintf('%11.5e',$spedi_RSC);
$RSC_str=~s/(e|E)([-+])/${2}0/;
$XNTOT_str=sprintf('%11.5e',$spedi_XNTOT);
$XNTOT_str=~s/(e|E)([-+])/${2}0/;

die("!!! The number of IMF exponents has to match the number of mass limits minus one !!!") 
  unless ($#IMF_m==$#IMF_a+1);

my @IMF_c=0*@IMF_a;
my $Mstar_avrg;
my $Mstar_avrg_IMF;
my $Mstar_avrg_MS;
my $Mmin_IMF;
my $Mmax_IMF;

init_MF();

print_info() unless ($spedi);

my $Mbin_inf=$IMF_m[0];
my $Mbin_sup;
my $Mavrg_bin;
my $Nfrac_bin;
my $Mfrac_bin;
my $lMinf=log($IMF_m[0]);
my $Minf=$IMF_m[0];
my $Mmax=($Mmax_IMF<$M_TO) ? $IMF_m[$#IMF_m] : $M_TO;
my $Dlm=log($Mmax/$Minf);
my $Dm=$Mmax-$Minf;
print "# 1: Mbin      2: Minf_ZAMS 3: N_frac    4: M_frac   5: Type\n" unless ($spedi);

if ($spedi) {
  #print "\n";
  #print "\n";
}

# Main Sequence
################

my $Type="MS";
my $iType=1;
for ($ibin=1; $ibin<=$Nbin; $ibin++) {
  if ($linear) {
    $Mbin_sup=$Minf+$Dm*($ibin/$Nbin)**$delta_discr;
  } else {
    $Mbin_sup=exp($lMinf+$Dlm*($ibin/$Nbin)**$delta_discr);
  }
  compute_and_write_component();
  $Mbin_inf=$Mbin_sup;
}

# White dwarfs
###############

if ($Mmax_IMF>$M_TO) {
  $Type="WD";
  $iType=3;
  $Mavrg_bin=$M_WD;
  $Mbin_inf=$M_TO;
  $Mbin_sup=($Mmax_IMF>$Mmax_WD) ? $Mmax_WD : $Mmax_IMF;
  &compute_and_write_component();
}

# Neutron stars
################

if ($Mmax_IMF>$Mmax_WD) {
  $ibin++;
  $Type="NS";
  $iType=4;
  $Mbin_inf=$Mbin_sup;
  $Mbin_sup=($Mmax_IMF>$Mmax_NS) ? $Mmax_NS : $Mmax_IMF;
  $Mavrg_bin=$M_NS;
  &compute_and_write_component();
}

# Black holes
###############

if ($Mmax_IMF>$Mmax_NS) {
  $ibin++;
  $Type="BH";
  $iType=5;
  $Mbin_inf=$Mbin_sup;
  $Mbin_sup=$Mmax_IMF;
  $Mavrg_bin=$M_BH;
  &compute_and_write_component();
}

if ($spedi) {
  print "XNTOTA= ",$XNTOT_str,"  XIMF= 0.00000+000 LEQ=  T\n";
  print "813,2,44,813,2,45,814,2,44,814,2,45,814,3,44,814,3,45,0,0\n";
  print "\n";
  print "\n";
  print_info();
}


#========== SUBROUTINES ==========

sub print_info { # print info...
  print "# IMF discretisation\n";
  print "# Command line: ",$command_line,"\n";
  print "# IMF limit masses : ",join("|",@IMF_m),"\n";
  print "# IMF exponents    : ",join("|",@IMF_a),"\n";
  print "# Stellar evol. parameters\n";
  print "# M_TO, M_WD, M_NS, M_BH, Mmax_WD, Mmax_NS : ",$M_TO," ",$M_WD," ",$M_NS," ",$M_BH," ",$Mmax_WD," ",$Mmax_NS,"\n";
  print "# MS Discretisation exponent : ",$delta_discr,"\n";
  print "# Number of MS bins          : ",$Nbin,"\n";
  print "# Average mass : ",sprintf("%12.5e",$Mstar_avrg),"\n";
  print "# Average MS mass : ",sprintf("%12.5e",$Mstar_avrg_MS),"\n";
  print "# Average IMF mass : ",sprintf("%12.5e",$Mstar_avrg_IMF),"\n";
}

sub init_MF { # compute continuity coefficients of IMF (normalized to 1)
  #print STDERR "> entering init_MF\n";

  $Mmin_IMF=$IMF_m[0];
  $Mmax_IMF=$IMF_m[$#IMF_m];
  $IMF_c[0]=1;
  for ($iIMF=1; $iIMF<$#IMF_m; $iIMF++) {
    $IMF_c[$iIMF]=$IMF_c[$iIMF-1]*$IMF_m[$iIMF]**($IMF_a[$iIMF]-$IMF_a[$iIMF-1]);
  }
  # normalisation
  my $norm=0;
  # Average mass on IMF
  $Mstar_avrg_IMF=0;

  for ($iIMF=0; $iIMF<$#IMF_m; $iIMF++) {
    $a1=1-$IMF_a[$iIMF];
    $norm=$norm            +$IMF_c[$iIMF]/$a1*($IMF_m[$iIMF+1]**$a1-$IMF_m[$iIMF]**$a1);
    $a2=2-$IMF_a[$iIMF];
    $Mstar_avrg_IMF=$Mstar_avrg_IMF+$IMF_c[$iIMF]/$a2*($IMF_m[$iIMF+1]**$a2-$IMF_m[$iIMF]**$a2);
  }
  $Mstar_avrg_IMF=$Mstar_avrg_IMF/$norm;
  for ($iIMF=0; $iIMF<=$#IMF_c; $iIMF++) {
    $IMF_c[$iIMF]=+$IMF_c[$iIMF]/$norm;
  }

  # Average mass after evolution
  $Mstar_avrg=0;
  my $Minf=0;
  my $Msup=0;
  my $Mdum=0;
  my $fracN=0;
  #print STDERR " Minf,Msup :",$Minf,",",$Msup,"\n";

  # MS stars
  $Minf=$Mmin_IMF;
  $Msup=$M_TO;

  #print STDERR " MS Minf,Msup :",$Minf,",",$Msup,"\n";
  if ($Minf<$Msup && $Msup<$Mmax_IMF) {
    ( $Mdum, $fracN ) = compute_bin($Minf,$Msup);
    $Mstar_avrg_MS = $Mdum;
    $Mstar_avrg = $Mstar_avrg+$fracN*$Mstar_avrg_MS;
  }

  # White dwarfs
  $Minf=$Msup;
  $Msup=($Mmax_IMF>$Mmax_WD) ? $Mmax_WD : $Mmax_IMF;
  #print STDERR " WD Minf,Msup :",$Minf,",",$Msup,"\n";
  if ($Minf>=$Mmin_IMF && $Msup>$Minf) {
    ( $Mdum, $fracN ) = compute_bin($Minf,$Msup);
    $Mstar_avrg = $Mstar_avrg+$fracN*$M_WD;
  }

  # Neutron stars
  $Minf=$Msup;
  $Msup=($Mmax_IMF>$Mmax_NS) ? $Mmax_NS : $Mmax_IMF;
  #print STDERR " NS Minf,Msup :",$Minf,",",$Msup,"\n";
  if ($Minf>=$Mmin_IMF && $Msup>$Minf) {
    ( $Mdum, $fracN ) = compute_bin($Minf,$Msup);
    $Mstar_avrg = $Mstar_avrg+$fracN*$M_NS;
  }

  # Black holes
  $Minf=$Msup;
  $Msup=$Mmax_IMF;
  #print STDERR " BH Minf,Msup :",$Minf,",",$Msup,"\n";
  if ($Minf>=$Mmin_IMF && $Msup>$Minf) {
    ( $Mdum, $fracN ) = compute_bin($Minf,$Msup);
    $Mstar_avrg = $Mstar_avrg+$fracN*$M_BH;
  }
  #print STDERR $Mstar_avrg," ",$Mstar_avrg_IMF,"\n";
  #print STDERR "< exiting init_MF\n";
}

sub compute_bin { # Compute average stellar mass and number fraction between $Mmin and $Mmax
  my ($Mmin, $Mmax) = @_;
  #print STDERR "> entering compute_bin ",$Mmin," ",$Mmax,"\n";
  return (0.5*($Mmin+$Mmax), 0.0) if ($Mmax<=$Mmin);
  return (0.5*($Mmin+$Mmax), 0.0) if ($Mmin>$IMF_m[$#IMF_m]);
  return (0.5*($Mmin+$Mmax), 0.0) if ($Mmax<$IMF_m[0]);

  my $iIMF=0;
  while ($IMF_m[$iIMF+1]<$Mmin) {
    $iIMF++;
  }

  # integrate
  my $Mslice=0.0;
  my $Nslice=0.0;
  my $Minf=0.0;
  my $Msup=0.0;
  while ($Msup<$Mmax && $iIMF<$#IMF_m) {
    $Minf=($Mmin>$IMF_m[$iIMF])   ? $Mmin : $IMF_m[$iIMF];
    $Msup=($Mmax<$IMF_m[$iIMF+1]) ? $Mmax : $IMF_m[$iIMF+1];
    $a2=2-$IMF_a[$iIMF];
    $Mslice=$Mslice+$IMF_c[$iIMF]/$a2*($Msup**$a2-$Minf**$a2);
    $a1=1-$IMF_a[$iIMF];
    $Nslice=$Nslice+$IMF_c[$iIMF]/$a1*($Msup**$a1-$Minf**$a1);
    $iIMF++;
  }
  #print STDERR $Mslice,"/",$Nslice,"\n";
  #print STDERR "> exiting compute_bin\n";
  return ($Mslice/$Nslice, $Nslice);
}

sub compute_and_write_component {
  my $m_dum;
  my $m_rem;
  my $itype;
  ( $m_dum, $Nfrac_bin ) = &compute_bin($Mbin_inf,$Mbin_sup);
  if ($iType==1) {
    $Mavrg_bin = $m_dum;
    ($itype, $m_rem) = &Mms_to_remnant($Mavrg_bin);
  } else {
    $m_rem = $Mavrg_bin;
  }
  $Mfrac_bin = $Nfrac_bin*$Mavrg_bin/$Mstar_avrg;
  $Rstel = &rel_MR($Mavrg_bin,$iType);
  if ($spedi) {
    $MIND_str=sprintf('%11.5e',$Mavrg_bin);
    $MIND_str=~s/(e|E)([-+])/${2}0/;
    $RIND_str=sprintf('%11.5e',$Rstel);
    $RIND_str=~s/(e|E)([-+])/${2}0/;
    $MREM_str=sprintf('%11.5e',$m_rem);
    $MREM_str=~s/(e|E)([-+])/${2}0/;
    $MTOT_str=sprintf('%11.5e',$Mavrg_bin*$Nfrac_bin*$spedi_XNTOT);
    $MTOT_str=~s/(e|E)([-+])/${2}0/;
    print sprintf('%2d',$ibin),"   MTOT= ",$MTOT_str," MIND= ",$MIND_str,"  MREM= ",$MREM_str," TAU= 1.00000+020\n";
    print "     RTOT= 2.23000+002 RIND= ",$RIND_str,"  TIND= 1.00000+000 RSC= ",$RSC_str,"\n";
    # May need to adapt value of XBIND to stellar type ???
    print "      EXP= 0.00000+000 UINI=-1.00000-005 XBIND= 1.00000+000 NOL= ",sprintf('%3d',$spedi_NOL),"\n";
    print "      LEQ=  T  T  T  T  T  T\n";
  } else {
    print sprintf(" %12.5e"x4,$Mavrg_bin,$Mbin_inf,$Nfrac_bin,$Mfrac_bin)," ",$Type,"\n";
  }
}

sub rel_MR {
  my ($mass, $type) = @_;
  open COMPUTE_MR, "echo $mass $type | rel_mr 2> /dev/null |";
  my $str = <COMPUTE_MR>;
  close COMPUTE_MR;
  # Trim leading spaces
  $str =~ s/^\s+//;
  my $crap;
  ($R,$crap) = split(/  */,$str,2);
  return $R;
}

sub Mms_to_remnant {
  my $Mms = $_[0];
  if ($Mms<$Mmax_WD) {
    return ( ($Mms<$M_WD) ? (iType_WD,$Mms) : (iType_WD,$M_WD) );
  } elsif ($Mms<$Mmax_NS) {
    return (iType_NS,$M_NS);
  } else {
    return (iType_BH,$M_BH);
  }
}
